# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

from django.core.handlers.wsgi import STATUS_CODE_TEXT
from django_yauth.middleware import (
    YandexAuthMiddleware, YandexAuthRequiredMiddleware
)

from travel.avia.avia_api.avia.middleware.base import WsgiMiddleware


log = logging.getLogger(__name__)


class WsgiYandexAuthMiddleware(WsgiMiddleware):
    """
    Обёртка над yauth.YandexAuthMiddleware
    Вытаскивает request.django_yauser
    и кладёт в environ -- ссылку на авторизованного
    пользователя Яндекса.
    """
    def __init__(self, app):
        super(WsgiYandexAuthMiddleware, self).__init__(app)
        self._django_middleware = YandexAuthMiddleware()

    def __call__(self, environ, start_response):
        self.process_environ(environ)
        return super(WsgiYandexAuthMiddleware).__call__(
            environ, start_response
        )

    def process_environ(self, environ):
        request = environ['django_request']
        response = self.process_request(request)

        log.debug('%s response: %r', self.__class__.__name__, response)

        if response is not None:
            raise response

        environ['django_yauser'] = request.yauser

    def process_request(self, request):
        return self._django_middleware.process_request(request)


class WsgiYandexAuthRequiredMiddleware(WsgiMiddleware):
    """
    Обёртка над yauth.YandexAuthRequiredMiddleware
    Вытаскивает request.django_yauser
    и кладёт в environ -- ссылку на авторизованного
    пользователя Яндекса.
    """
    def __init__(self, app):
        super(WsgiYandexAuthRequiredMiddleware).__init__(app)
        self._django_middleware = YandexAuthRequiredMiddleware()

    def __call__(self, environ, start_response):
        with self.measure():
            response = self.process_environ(environ)
            log.debug('%s response: %r', self.__class__.__name__, response)

            if response is not None:
                try:
                    status_text = STATUS_CODE_TEXT[response.status_code]
                except KeyError:
                    status_text = 'UNKNOWN STATUS CODE'
                status = '%s %s' % (response.status_code, status_text)
                response_headers = [
                    (str(k), str(v)) for k, v in response.items()
                ]
                for c in response.cookies.values():
                    response_headers.append(
                        ('Set-Cookie', str(c.output(header='')))
                    )
                start_response(status, response_headers)
                return response

        return super(WsgiYandexAuthRequiredMiddleware).__call__(
            environ, start_response
        )

    def process_environ(self, environ):
        request = environ['django_request']
        return self.process_request(request)

    def process_request(self, request):
        return self._django_middleware.process_request(request)
