# -*- coding: utf-8 -*-
import logging
from contextlib import contextmanager

import travel.avia.contrib.python.mongoengine.mongoengine as mongoengine
import ujson
from enum import Enum
from flask import Blueprint
from travel.avia.contrib.python.mongoengine.mongoengine import DoesNotExist
from werkzeug.exceptions import Forbidden, NotImplemented

import travel.avia.avia_api.ant.api_interface
from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.ant.custom_types import Int, Str
from travel.avia.avia_api.ant.exceptions import ValidationError
from travel.avia.avia_api.avia.lib.passport_utils import get_email_by_uid, get_emails_by_uid
from travel.avia.avia_api.avia.lib.yt_loggers.email_subscriptions.email_logs import (
    log_1_opt_in_email_sent, log_2_opt_in_email_sent,
)
from travel.avia.avia_api.avia.lib.yt_loggers.email_subscriptions.user_logs import (
    log_user_confirm, log_user_subscribed,
    log_user_unsubscribe, log_user_unsubscribe_direction,
)
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.build_email import EMAIL_DATE_FORMAT
from travel.avia.avia_api.avia.v1.email_dispenser.sender import TransactionalApi
from travel.avia.avia_api.avia.v1.email_dispenser.settings import PriceChangesSettings
from travel.avia.avia_api.avia.v1.model.filters import Filter
from travel.avia.avia_api.avia.v1.model.subscriber import MinPrice, Subscriber, Subscription, hashed

email_subscription_blueprint = Blueprint('email_subscription_v1.1', __name__)

email_subscription_api = travel.avia.avia_api.ant.api_interface.Ant(email_subscription_blueprint)

log = logging.getLogger(__name__)

_single_opt_in = TransactionalApi(PriceChangesSettings, PriceChangesSettings.single_opt_in_campaign_slug)
_double_opt_in = TransactionalApi(PriceChangesSettings, PriceChangesSettings.double_opt_in_campaign_slug)


class OptinType(Enum):
    """
    Enum типа подтверждения подписки
    """
    single = 1
    double = 2
    pending = 3


def qkey_from_qid(qid):
    """
    :param basestring qid:
    :rtype: basestring
    """
    name_parts = qid.split('.')
    return name_parts[3]


def _validate_qkey(qid, qkey):
    """
    :param basestring qid:
    :param basestring qkey:
    :rtype: basestring
    """
    if not qkey:
        if qid:
            qkey = qkey_from_qid(qid)
        else:
            raise ValidationError('qid and qkey are not specified', fields=['qid', 'qkey'])
    else:
        if '.' in qkey or '$' in qkey:
            raise ValidationError('qkey is invalid (should not contain "." or "$")', fields=['qkey'])
    return qkey


def get_readable_flight_info(qkey):
    """
    :param basestring qkey: идентификатор перелета
    :return: человекочитаемые данные о перелете
    """
    s = Subscription.from_qkey(qkey)
    return {
        'city_from': unicode(s.point_from.title) if s.point_from else None,
        'city_to': unicode(s.point_to.title) if s.point_to else None,
        'date_forward': s.date_forward.strftime(EMAIL_DATE_FORMAT) if s.date_forward else None,
        'date_backward': s.date_backward.strftime(EMAIL_DATE_FORMAT) if s.date_backward else None,
        'avia_search_link': s.avia_search_link()
    }


def normalize_email(email):
    """
    :param basestring email:
    :rtype: basestring
    """
    try:
        return email.strip().lower()
    except AttributeError:
        return email


def send_1opt_in(subscriber, qkey):
    """
    Вместо 2opt-in подтверждения подписки просто отправляем информационное письмо
    :param avia.v1.model.subscriber.Subscriber subscriber:
    :param basestring qkey:
    """
    args = {
        'unsubscribe_url': subscriber.unsubscribe_link(),
    }
    args.update(get_readable_flight_info(qkey))
    try:
        _single_opt_in.send(subscriber.email, async_send=False, args=args)
        log_1_opt_in_email_sent(email=hashed(subscriber.email), qkey=qkey, email_content=args)
        log.info(
            'Successfully sent 1opt-in to %s for qkey %s',
            subscriber.email,
            qkey,
        )
    except Exception:
        log.exception('Could not send 1-opt-in to %s with arguments %s', subscriber.email, args)


def send_2opt_in(subscriber, qkey, passport_plain=None, session_plain=None, filter_fragment=None):
    """
    :param avia.v1.model.subscriber.Subscriber subscriber:
    :param basestring qkey:
    :param basestring | None passport_plain:
    :param basestring | None session_plain:
    :param basestring | None filter_fragment: string containing filter in frontend format (ex.: "#bg=1")
    :rtype: bool
    """
    if not PriceChangesSettings.double_opt_in_campaign_slug:
        raise NotImplemented('Missing 2opt-in template slug')  # noqa: F901
    args = {
        'subscribe_link': subscriber.confirm_link(passport_plain, session_plain, qkey, filter_fragment=filter_fragment),
    }
    args.update(get_readable_flight_info(qkey))
    try:
        _double_opt_in.send(subscriber.email, async_send=False, args=args)
        log_2_opt_in_email_sent(email=hashed(subscriber.email), qkey=qkey, email_content=args)
        log.info(
            'Successfully sent 2opt-in to %s for qkey %s',
            subscriber.email,
            qkey,
        )
        return True
    except Exception:
        log.exception('Could not send 2-opt-in to %s with arguments %s', subscriber.email, args)
        return False


@contextmanager
def db_subscriber(create=False, **kwargs):
    """
    :param bool create:
    :param kwargs:
    :return:
    """
    try:
        subscriber = Subscriber.objects.get(**kwargs)
        if subscriber.api_version != '1.1':
            log.debug('Updating api version')
            update_api_version(subscriber)
    except DoesNotExist:
        log.info('Subscriber for {} params doesn\'t exists'.format(kwargs))
        if create:
            subscriber = Subscriber(api_version='1.1', **kwargs)
            log.info('Create new subscriber for {}'.format(kwargs))
        else:
            raise
    yield subscriber
    subscriber.save()


def update_api_version(subscriber):
    """
    :param avia.v1.model.subscriber.Subscriber subscriber:
    :return:
    """
    log.debug('Updating API version')
    for qkey, subscription in subscriber.subscriptions.iteritems():
        subscription.approve()
        subscription.update_pending()
        subscription.update_source(subscription.source, force=True)
    subscriber.api_version = '1.1'


def _select_optin_type(subscriber, passport_plain, session_plain):
    """
    Логика выбора между 1opt-in и 2opt-in для пользователя

    :param basestring passport_plain: пасспортный uid залогиненого пользователя
    :param basestring session_plain: id сессии, с которой пришел незалогиненый пользователь
    :param avia.v1.model.subscriber.Subscriber subscriber: список сессий подписчика
    :return: Тип подтверждения 1opt-in или 2opt-in
    :rtype: OptinType
    """
    if passport_plain:
        subscriber.cleanup_passports()
        passport_emails = map(normalize_email, get_emails_by_uid(passport_plain))
        log.debug(
            'Subscriber %s has %s emails linked to passport %s',
            subscriber,
            len(passport_emails),
            passport_plain,
        )
        if passport_emails and subscriber.email in passport_emails:
            return OptinType.single
        elif subscriber.is_2opt_in_progress(passport_hashed=hashed(passport_plain)):
            return OptinType.pending
        elif hashed(passport_plain) in subscriber.related_passports:
            return OptinType.single
        else:
            return OptinType.single  # OptinType.double before RASPTICKETS-19061
    elif session_plain:
        subscriber.cleanup_sessions()
        if subscriber.has_too_many_unapproved_sessions():
            log.debug('subscriber %s with too many unapproved sessions', subscriber)
            raise Forbidden([{
                'errors': {
                    'error': 'Too many sessions',
                    'ban_until': subscriber.session_unban_time()
                },
            }])
        elif subscriber.is_2opt_in_progress(session_hashed=hashed(session_plain)):
            return OptinType.pending
        elif hashed(session_plain) in subscriber.related_sessions:
            return OptinType.single
        else:
            return OptinType.single  # OptinType.double before RASPTICKETS-19061
    else:
        return None


def subscribe(
    subscriber, qkey, source,
    filter_=None, pending_passport_plain=None, pending_session_plain=None,
    date_range=1, min_price=None
):
    """
    :param avia.v1.model.subscriber.Subscriber subscriber:
    :param basestring qkey:
    :param basestring source:
    :param avia.v1.model.filters.Filter filter_:
    :param basestring pending_passport_plain: plaintext passport
    :param basestring pending_session_plain: plaintext session
    :param int date_range: range of subscription dates
    :param MinPrice min_price : a price that user saw during subscription
    :return:
    """
    qkeys = subscriber.add_subscription(
        qkey,
        source,
        filter_=filter_,
        pending_passport_plain=pending_passport_plain,
        pending_session_plain=pending_session_plain,
        date_range=date_range,
        min_price=min_price,
    )
    if len(qkeys) > 1:
        log.info('Adding complex subscription (%s) for subscriber %s', len(qkeys), subscriber.email)
    for qk in qkeys:
        try:
            subscription = Subscription.objects.get(qkey=qk)
        except DoesNotExist:
            subscription = Subscription.from_qkey(qk)
            log.info('Adding subscription %s to collection', qk)

        subscription.add_filter(filter_)
        subscription.save()
    log_user_subscribed(
        email=hashed(subscriber.email),
        qkey=qkey,
        source=source,
        filter_params=ujson.loads(filter_.to_json()) if filter_ else None,
        pending_passport=hashed(pending_passport_plain),
        pending_session=hashed(pending_session_plain),
        date_range=date_range,
        min_price=ujson.loads(min_price.to_json()) if min_price else None,
    )
    log.info(
        'Added subscription %s to subscriber %s',
        qkey,
        subscriber.email,
    )


@email_subscription_api.view('/subscribe')
def email_subscribe_handler(
    passport_plain=ViewParam(name='passportuid', type_=Str(), required=False),
    session_plain=ViewParam(name='session', type_=Str(), required=False),
    email=ViewParam(type_=Str(), required=False),
    qid=ViewParam(type_=Str(), required=False),
    qkey=ViewParam(type_=Str(), required=False),
    email_source=ViewParam(type_=Str(), required=True),
    filter_=ViewParam(name='filter', type_=Str(), required=False),
    date_range=ViewParam(type_=Int(interval=(1, 120)), default=1, required=False),
    min_price=ViewParam(type_=Str(), required=False),
):
    return _email_subscribe_handler(
        passport_plain=passport_plain,
        session_plain=session_plain,
        email=email,
        qid=qid,
        qkey=qkey,
        email_source=email_source,
        filter_=filter_,
        date_range=int(date_range),
        min_price=min_price,
    )


def _email_subscribe_handler(
    passport_plain=None,
    session_plain=None,
    email=None,
    qid=None,
    qkey=None,
    email_source=None,
    filter_=None,
    date_range=1,
    min_price=None,
):
    if not passport_plain and not session_plain:
        raise ValidationError('user without session and passport')
    if not passport_plain and not email:
        raise ValidationError('user without email and passport')
    qkey = _validate_qkey(qid, qkey)

    if passport_plain and not email:
        email = get_email_by_uid(passport_plain)
        if not email:
            raise ValidationError('User did not provide email and has no email linked to passport')

    email = normalize_email(email)
    subscriber_info = {'email': email}
    log.debug(
        'User with email %s, passport %s and/or session %s',
        email,
        passport_plain,
        session_plain,
    )
    try:
        filter_obj = Filter.from_json(filter_) if filter_ else None
    except ValueError:
        log.warning('Unable to decode filter from json %s', filter_)
        filter_obj = None

    try:
        min_price_obj = MinPrice.from_json(min_price) if min_price else None  # type: MinPrice | None
        if min_price_obj:
            min_price_obj.validate()
    except (ValueError, mongoengine.errors.ValidationError):
        log.warning('Unable to decode minprice from json %s', min_price)
        min_price_obj = None

    with db_subscriber(create=True, **subscriber_info) as subscriber:
        log.debug('User %s found or created', subscriber_info)
        optin_type = _select_optin_type(subscriber, passport_plain, session_plain)
        if optin_type is None:
            raise ValidationError('Error detecting optin type')

        log.debug(
            'Choosing optin %s for User with email %s, passport %s and/or session %s',
            optin_type,
            email,
            passport_plain,
            session_plain,
        )
        if optin_type == OptinType.single:
            subscribe(
                subscriber,
                qkey,
                email_source,
                filter_=filter_obj,
                date_range=date_range,
                min_price=min_price_obj,
            )
            subscriber.request_relation(passport_plain=passport_plain, session_plain=session_plain)
            subscriber.approve_relation(passport_hashed=hashed(passport_plain), session_hashed=hashed(session_plain))
            send_1opt_in(subscriber, qkey)
            log.info(
                '1OPT User %s email %s subscribed to %s from %s with passport=%s or session=%s',
                subscriber.id,
                subscriber.email,
                qkey,
                email_source,
                hashed(passport_plain),
                hashed(session_plain),
            )
        elif optin_type == OptinType.double:
            if send_2opt_in(
                subscriber=subscriber,
                qkey=qkey,
                passport_plain=passport_plain,
                session_plain=session_plain,
                filter_fragment=filter_obj.frontend_filter_postfix if filter_obj else None
            ):
                subscriber.request_relation(passport_plain=passport_plain, session_plain=session_plain)
                subscribe(
                    subscriber,
                    qkey,
                    email_source,
                    filter_=filter_obj,
                    pending_passport_plain=passport_plain,
                    pending_session_plain=session_plain,
                    date_range=date_range,
                    min_price=min_price_obj,
                )
                log.info(
                    '2OPT User %s email %s subscribed to %s from %s with passport=%s or session=%s',
                    subscriber.id,
                    subscriber.email,
                    qkey,
                    email_source,
                    hashed(passport_plain),
                    hashed(session_plain),
                )
            else:
                log.warning(
                    'Unable to send 2opt-in message to %s with qkey %s',
                    subscriber.email,
                    qkey,
                )
        elif optin_type == OptinType.pending:
            subscriber.request_relation(passport_plain=passport_plain, session_plain=session_plain)
            subscribe(
                subscriber,
                qkey,
                email_source,
                filter_=filter_obj,
                pending_passport_plain=passport_plain,
                pending_session_plain=session_plain,
                date_range=date_range,
                min_price=min_price_obj,
            )
            log.info(
                '2OPT-pending User %s email %s subscribed to %s from %s with passport=%s or session=%s',
                subscriber.id,
                subscriber.email,
                qkey,
                email_source,
                hashed(passport_plain),
                hashed(session_plain),
            )
    log.debug('Done with susbscriber in /subscribe handle')
    return {
        'id': subscriber and subscriber.id,
        'pending': optin_type == OptinType.pending,
        'required_double_opt_in': optin_type == OptinType.pending or optin_type == OptinType.double
    }


@email_subscription_api.view('/unsubscribe')
def email_unsubscribe_handler(
    _id=ViewParam(name='id', type_=Str()),
):
    with db_subscriber(id=_id) as subscriber:
        subscriber.subscriptions = {}
        log_user_unsubscribe(
            email=hashed(subscriber.email),
        )
        log.info('User %s with email %s unsubscribed from all', _id, subscriber.email)

    return {
        'id': subscriber and subscriber.id
    }


@email_subscription_api.view('/unsubscribe/by_direction')
def email_unsubscribe_by_direction_handler(
    _id=ViewParam(name='id', type_=Str()),
    qkey=ViewParam(type_=Str()),
):
    with db_subscriber(id=_id) as subscriber:
        deleted_subscription = subscriber.subscriptions.pop(qkey, None)
        if deleted_subscription:
            filters = deleted_subscription.applied_filters
            filter_ = ujson.loads(filters[0].to_json()) if filters else None
            log_user_unsubscribe_direction(
                email=hashed(subscriber.email),
                qkey=str(qkey),
                date_range=deleted_subscription.date_range,
                filter_params=filter_,
                pending_passport=deleted_subscription.pending_passport,
                pending_session=deleted_subscription.pending_session,
            )
        log.info('User %s with email %s unsubscribed from qkey: %s', _id, subscriber.email, qkey)

    return {
        'id': subscriber and subscriber.id
    }


@email_subscription_api.view('/confirm')
def double_opt_confirm_handler(
    _id=ViewParam(name='id', type_=Str(), required=True),
    passport_hashed=ViewParam(name='p', type_=Str(), required=False),
    session_hashed=ViewParam(name='s', type_=Str(), required=False),
):
    return _double_opt_confirm_handler(
        _id,
        passport_hashed,
        session_hashed,
    )


def _double_opt_confirm_handler(
    _id=None,
    passport_hashed=None,
    session_hashed=None,
):
    """
    Привязываем пользователя к email по cессии или по passport
    Включаем накопленные подписки
    :return:
    """
    log.info('User %s tries to confirm passport=%s or session=%s', _id, passport_hashed, session_hashed)
    if not passport_hashed and not session_hashed:
        raise ValidationError('no passport and session provided for approval')
    confirmed = False
    with db_subscriber(id=_id) as subscriber:
        if subscriber.approve_relation(passport_hashed=passport_hashed, session_hashed=session_hashed):
            confirmed = True
            pending = subscriber.get_pending_subscriptions_list(
                passport_hashed=passport_hashed,
                session_hashed=session_hashed
            )
            for qkey in pending:
                subscriber.subscriptions[qkey].approve()
                filters = subscriber.subscriptions[qkey].applied_filters
                filter_ = ujson.loads(filters[0].to_json()) if filters else None
                log_user_confirm(
                    email=hashed(subscriber.email),
                    qkey=qkey,
                    date_range=subscriber.subscriptions[qkey].date_range,
                    filter_params=filter_,
                    pending_passport=passport_hashed,
                    pending_session=session_hashed,
                )
                send_1opt_in(subscriber, qkey)
            log.info(
                'User %s with email %s succesfully confirmed passport=%s or session=%s and pending subscriptions %s',
                _id,
                subscriber.email,
                passport_hashed,
                session_hashed,
                pending,
            )
        else:
            log.info(
                'User %s with email %s was not able to confirm passport=%s or session=%s',
                _id,
                subscriber.email,
                passport_hashed,
                session_hashed,
            )

    return {
        'id': subscriber and subscriber.id,
        'confirmed': confirmed,
    }


@email_subscription_api.view('/user_subscriptions/list')
def user_subscriptions_list(
    passport_plain=ViewParam(name='passportuid', required=False, type_=Str()),
    email=ViewParam(name='email', required=False, type_=Str()),
):
    """ Получить подписки списком """
    if not passport_plain and not email:
        raise ValidationError('Provide passportuid or email to get list of subscriptions')
    if passport_plain and not email:
        email = get_email_by_uid(passport_plain)
    email = normalize_email(email)
    subscriber = None
    try:
        log.debug('About to find a subscriber in db')
        subscriber = Subscriber.objects.get(email=email)
        log.debug('subscriptions: %s', subscriber.subscriptions)
        subscriptions = subscriber.get_approved_subscriptions_list()
    except DoesNotExist:
        subscriptions = []
    return {
        'subscriptions': subscriptions,
        'id': subscriber and subscriber.id
    }


@email_subscription_api.view('/user_subscriptions')
def user_subscriptions(
    passport_plain=ViewParam(name='passportuid', required=False, type_=Str()),
    email=ViewParam(type_=Str(), required=False),
    qid=ViewParam(type_=Str(), required=False),
    qkey=ViewParam(type_=Str(), required=False),
):
    """ Проверить, подписан ли пользователь на направление """
    if not passport_plain and not email:
        raise ValidationError('Provide passportuid or email to see if user is subscribed or not')
    if passport_plain and not email:
        email = get_email_by_uid(passport_plain)
    email = normalize_email(email)
    qkey = _validate_qkey(qid, qkey)
    subscriber = None
    filter_ = None
    original_qkey = None
    date_range = None
    try:
        subscriber = Subscriber.objects.get(email=email)
        subscriber_expanded_subscriptions = subscriber.expand_approved_subscriptions()
        subscribed = qkey in subscriber_expanded_subscriptions
        if subscribed:
            original_qkey = subscriber_expanded_subscriptions[qkey]
            subscriber_subscription = subscriber.subscriptions[original_qkey]
            date_range = subscriber_subscription.date_range
            filter_ = (
                subscriber_subscription.applied_filters[0].frontend_filter_postfix
                if subscriber_subscription.applied_filters else None
            )
    except DoesNotExist:
        subscribed = False

    return {
        'subscribed': subscribed,
        'id': subscriber and subscriber.id,
        'filter': filter_,
        'qkey': original_qkey,
        'date_range': date_range,
    }
