# -*- coding: utf-8 -*-
import logging
import time
from datetime import datetime, timedelta

from django.conf import settings
from yt.wrapper import YtClient

from travel.avia.library.python.common.models.geo import Country

from travel.avia.avia_api.avia.daemon_http_api import DaemonBadRequest, TicketDaemon
from travel.avia.avia_api.avia.v1.email_dispenser.commands.decorators import stdout_logger
from travel.avia.avia_api.avia.v1.email_dispenser.commands.update_subscriptions_info import (
    actualize_subscriptions, relevant_subscriptions,
)
from travel.avia.avia_api.avia.v1.email_dispenser.variants_fetcher.preheat_subscriptions_yt import filter_cold_qkey
from travel.avia.avia_api.avia.v1.model.distributed_lock import single_process
from travel.avia.avia_api.avia.v1.model.subscriber import Subscription
from travel.avia.avia_api.avia.v1.model.variants_compute_history import SubscriptionPreheatHistory

log = logging.getLogger(__name__)

INIT_RPS = 2.0
INIT_REQUEST_DELAY = 1.0 / INIT_RPS


@stdout_logger
@single_process('preheat_subscriptions', datetime.utcnow() + timedelta(minutes=30), release=False)
def preheat_subscriptions():
    ytc = YtClient(
        token=settings.YT_TOKEN,
        proxy=settings.YT_PROXY,
        config={
            'remote_temp_tables_directory': '//home/avia/tmp',
        },
    )
    log.info("Actualizing subscriptions")
    actualize_subscriptions()
    log.info("Preheating relevant subscriptions")
    active_qkeys = {sub.qkey for sub in relevant_subscriptions()}

    try:
        cold_qkeys = filter_cold_qkey(active_qkeys, ytc, log)
    except Exception:
        log.exception('YT filtering error')
        return

    qids = list()
    log.info('Going to preheat %d qkeys with %f RPS', len(cold_qkeys), INIT_RPS)
    limit_exceeded_flag = False
    try:
        for qkey in cold_qkeys:
            time.sleep(INIT_REQUEST_DELAY)
            subscription = Subscription.from_qkey(qkey)
            log.info('TicketDaemon init_search %s', qkey)
            search_params = {
                'point_from': subscription.point_from,
                'point_to': subscription.point_to,
                'date_forward': subscription.date_forward,
                'date_backward': subscription.date_backward,
                'adults': subscription.passengers.adults,
                'children': subscription.passengers.children,
                'infants': subscription.passengers.infants,
                'flight_class': subscription.klass,
                'national_version': subscription.national_version,
                'lang': subscription.lang,
                'country_geo_id': Country.RUSSIA_GEO_ID,
                'service': 'email_subscriptions'
            }
            try:
                qid = TicketDaemon(settings.DAEMON_URL).init_search(**search_params)
            except DaemonBadRequest as de:
                if 'Limits exceeded' in de.message:
                    log.exception('Limit rate exceeded, stopping heater')
                    limit_exceeded_flag = True
                else:
                    log.exception('Daemon bad request')
                break
            except Exception:
                log.exception(search_params)
                # Разгружаем демона
                time.sleep(1.0)
                continue
            qids.append(qid)
    except Exception:
        log.exception('Error during preheat process')

    history_data = {
        'active_subscriptions_count': len(active_qkeys),
        'cold_subscriptions_count': len(cold_qkeys),
        'preheated_subscriptions_count': len(qids),
        'td_limit_exceeded': limit_exceeded_flag,
    }
    log.info('Saving data for history: %s', history_data)
    SubscriptionPreheatHistory(**history_data).save()
    return qids
