# -*- coding: utf-8 -*-
import logging
from collections import defaultdict

import ujson
from datetime import datetime, timedelta
from django.conf import settings
from travel.avia.contrib.python.mongoengine.mongoengine import DoesNotExist
from yt.wrapper import YtClient

from travel.avia.avia_api.avia.lib.mongo_iterator import iterate_chunks
from travel.avia.avia_api.avia.lib.yt_loggers.email_subscriptions.email_logs import log_price_change_email_part
from travel.avia.avia_api.avia.v1.email_dispenser.commands.decorators import stdout_logger
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.build_email import (
    LetterBlockBuilder, _format_date, format_price_diff_value_html,
    format_price_value_html,
)
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.currency import currency_to_symbol
from travel.avia.avia_api.avia.v1.email_dispenser.sender import TransactionalApi
from travel.avia.avia_api.avia.v1.email_dispenser.settings import PriceChangesSettings
from travel.avia.avia_api.avia.v1.model.distributed_lock import single_process
from travel.avia.avia_api.avia.v1.model.subscriber import Subscriber, SubscriberSubscription, Subscription, hashed

log = logging.getLogger(__name__)

_extended_letter_campaign = TransactionalApi(PriceChangesSettings, PriceChangesSettings.campaign_slug_extended)


@stdout_logger
def send_email_to(email):
    _send_emails(email)


def send_emails():
    try:
        return single_process(
            'send_emails', datetime.utcnow() + timedelta(minutes=10), release=False
        )(_send_emails)()
    except:
        log.exception('Got an error while sending subscriptions')


def get_letter_builder(_logger):
    ytc = YtClient(
        proxy=settings.YT_PROXY,
        token=settings.YT_TOKEN,
        config={
            'remote_temp_tables_directory': '//home/avia/tmp',
        },
    )
    return LetterBlockBuilder(yt=ytc, logger=_logger)


def _send_emails(email=None):
    log.info('Start sending emails')
    log.info('%d subscriptions', Subscription.objects.count())
    log.info('%d subscribers', Subscriber.objects.count())

    if email:
        try:
            subscribers_query = Subscriber.objects.filter(email=email)
        except DoesNotExist:
            log.info("No subscribers with email %s found", email)
            return
    else:
        subscribers_query = Subscriber.objects.all()

    sent_subscriptions_price = defaultdict(dict)
    for subscriber in iterate_chunks(Subscriber, 'id', subscribers_query):
        log.info('Looking at subscriber %r', subscriber)
        if not subscriber.subscribed:
            continue
        try:
            email_content = build_extended_email_content(
                subscriber=subscriber,
                letter_builder=get_letter_builder(log),
            )
        except Exception:
            log.exception('Failed to build email for %r', subscriber)
            continue
        if email_content:
            log.info('Sending email to %r', subscriber)
            args = {
                'content': email_content,  # todo: Не нужно отсортировать этот список? Например, свежие подписки вверху?
                'unsubscribe_url': subscriber.unsubscribe_link()
            }
            log.debug('Email args %r', args)
            try:
                _extended_letter_campaign.send(subscriber.email, async_send=False, args=args)
            except Exception:
                log.exception('Failed to send email to %r', subscriber)
    log.info('Emails are sent')

    if not email and sent_subscriptions_price:
        log.info('Start updating %d Subscriptions last sent MinPrice', len(sent_subscriptions_price))
        for subscription in Subscription.objects.filter(qkey__in=sent_subscriptions_price.keys()):
            for applied_filter, min_price in sent_subscriptions_price[subscription.qkey].iteritems():
                if applied_filter is None:
                    subscription.modify(set__sent_min_price=min_price)
                else:
                    subscription.modify(
                        query={'filtered_minprices__filter': applied_filter},
                        set__filtered_minprices__S__sent_min_price=min_price
                    )
        log.info('Subscriptions last sent MinPrice are updated')

    print('Sent emails to subscribers. See log for details.')


def build_extended_email_content(subscriber, letter_builder):
    """
    :param typing.Dict[basestring, typing.Dict[avia.v1.model.filters.Filter|None, avia.v1.model.subscriber.MinPrice]] sent_subscriptions_price:
        Минцены подписок, отправленные пользователям.
    :param Subscriber subscriber:
    :param LetterBlockBuilder letter_builder:
    :return:
    """
    subscriptions = {
        s.qkey: s for s in
        Subscription.objects.filter(qkey__in=subscriber.expand_approved_subscriptions().keys())
    }
    subscriber.actualize_subscriptions(set(subscriptions.keys()))

    log.info('Preparing email for %r %s', subscriber, subscriptions)
    email_content = []
    for sqkey in subscriber.get_approved_subscriptions_list():
        log.info('Have approved %s', sqkey)

        if sqkey not in subscriber.subscriptions:
            logging.error(
                'Subscriber %s had subscription %s and now it\'s gone',
                subscriber, sqkey,
            )
            continue
        subscriber_subscription = subscriber.subscriptions[sqkey]  # type: SubscriberSubscription
        if subscriber_subscription is None:
            logging.error(
                'Subscriber %s subscription %s is suddenly None',
                subscriber, sqkey,
            )
            continue

        try:
            date_range = subscriber_subscription.date_range
        except (AttributeError, KeyError):
            log.warning(
                'Cannot get date range for subscriber %r subscription %r',
                subscriber,
                sqkey,
            )
            continue

        qkeys = subscriber.expand_qkey_daterange(sqkey, date_range)

        applied_filters = list(subscriber_subscription.applied_filters)
        if not applied_filters:
            applied_filters = [None]
        log.info('applied filters: %s', applied_filters)
        for applied_filter in applied_filters:
            qkey = find_min_price_qkey_in_date_range(qkeys, applied_filter)

            if qkey is None:
                log.warning(
                    'Cannot find relevant information about min price sqkey=%s, qkeys=%s, filter=%s',
                    sqkey,
                    qkeys,
                    applied_filter,
                )
                continue

            if qkey not in subscriptions:
                log.error('Subscription %s not found in subscriptions list %s', qkey)
                continue
            subscription = subscriptions[qkey]

            prev, now = subscription.last_filtered_min_price(applied_filter)
            try:
                if subscriber_subscription.last_seen_min_price is not None:
                    prev = subscriber_subscription.last_seen_min_price
                    log.info(
                        'Subscriber %s subscription %s last_seen_min_price = %s',
                        subscriber, sqkey, prev,
                    )
                else:
                    log.info(
                        'Subscriber %s subscription %s last_seen_min_price not found',
                        subscriber, sqkey,
                    )
            except AttributeError:
                log.exception(
                    'No last_seen_min_price field found for subscriber %s subscription %s',
                    subscriber, sqkey,
                )
                continue
            log.info('Filter %s, prev: %s, now %s', applied_filter, prev, now)

            try:
                # Минимальная цена на билет изменилась больше чем на 3% с момента последней отправки или за 24 часа
                if prev and now and abs(1 - now.value / prev.value) > 0.03:

                    price_diff = now.value - prev.value
                    content = {
                        'diff': price_diff,
                        'diff_str': format_price_diff_value_html(price_diff),
                        'point_from': subscription.point_from.title,
                        'point_to': subscription.point_to.title,
                        'date_forward': _format_date(subscription.date_forward),
                        'price': format_price_value_html(now.value),
                        'currency': currency_to_symbol(now.currency),
                        'old_price': format_price_value_html(prev.value),
                        'old_currency': currency_to_symbol(prev.currency),
                        'unsubscribe_by_direction_link': subscriber.unsubscribe_by_direction_link(sqkey),
                        'avia_search_link': subscription.avia_search_link(
                            filter_fragment=applied_filter.frontend_filter_postfix if applied_filter else None,
                        ),  # Ссылка на поиск
                    }

                    if subscription.date_backward:
                        content['date_backward'] = _format_date(subscription.date_backward)

                    if letter_builder:
                        variants_data = letter_builder.build_for_subscription(
                            subscription=subscription,
                            variants=now.variants,
                            applied_filter=applied_filter,
                            price_difference=price_diff,
                        )
                        if variants_data:
                            content.update(variants_data)

                    if len(qkeys) > 1:
                        price_dynamics = get_price_dynamics(qkeys, applied_filter)
                        content.update(price_dynamics)

                    email_content.append(content)
                    log_price_change_email_part(
                        email=hashed(subscriber.email),
                        qkey=sqkey,
                        date_range=date_range,
                        filter_params=ujson.loads(applied_filter.to_json()) if applied_filter else None,
                        email_content=content,
                    )
                    subscriber.modify(
                        **{
                            'set__subscriptions__{}__last_seen_min_price'.format(sqkey): now,
                        }
                    )

            except (AttributeError, KeyError, TypeError):
                log.exception(
                    'Cannot get data for subscription qkey = %s, sqkey = %s',
                    qkey, sqkey,
                )

    log.info('Email content %s', email_content)
    return email_content


def find_min_price_qkey_in_date_range(qkeys, applied_filter):
    """
    :param typing.Iterable qkeys:
    :param avia.v1.model.filters.Filter | None applied_filter:
    :return:
    """
    subscriptions = Subscription.objects(qkey__in=qkeys)
    min_price_value = None
    min_price_qkey = None
    for subscription in subscriptions:
        _, now_min_price = subscription.last_filtered_min_price(applied_filter)
        if not now_min_price or not now_min_price.value:
            continue
        if min_price_qkey is None or min_price_value is None or now_min_price.value < min_price_value.value:
            min_price_value = now_min_price
            min_price_qkey = subscription.qkey
    return min_price_qkey


def get_price_dynamics(qkeys, applied_filter):
    """
    {
        'price_dynamics': [
            {
                'date_forward':'2019-09-01',
                'value': 123,
                'currency': 'RUR'
            },
        ]
    }
    """
    dynamics = []
    subscriptions = Subscription.objects(qkey__in=qkeys)
    for subscription in subscriptions:
        date_forward = subscription.date_forward
        if not date_forward:
            continue
        min_price = subscription.last_filtered_min_price(applied_filter)[0]
        dynamics.append({
            'date_forward': _format_date(date_forward),
            'value': format_price_value_html(min_price.value) if min_price and min_price.value else None,
            'currency': currency_to_symbol(min_price.currency) if min_price and min_price.currency else None
        })

    dynamics.sort(key=lambda d: d['date_forward'])
    return {'price_dynamics': dynamics}
