# coding=utf-8
import logging
from datetime import date

from travel.avia.avia_api.avia.lib.mongo_iterator import iterate_chunks
from travel.avia.avia_api.avia.v1.email_dispenser.commands.decorators import stdout_logger
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.qkey_utils import structure_from_qkey
from travel.avia.avia_api.avia.v1.model.subscriber import Subscription

log = logging.getLogger(__name__)

today = date.today()


@stdout_logger
def cleanup_bad_subscriptions():
    log.info('Cleaning up bad subscriptions')
    invalid_subscriptions = set()
    log.info('Preparing')
    total = Subscription.objects.count()
    prepared = 0
    showed = 0.
    for subscription in iterate_chunks(Subscription, 'qkey', Subscription.objects.all().no_cache()):
        # check base subscription structure
        if (
            subscription.date_forward is None
            or subscription.point_from_key is None
            or subscription.point_to_key is None
        ):
            invalid_subscriptions.add(subscription.qkey)
        else:
            # check min prices
            qkey = subscription.qkey
            try:
                changes = False
                new_mp = []
                for mp in subscription.min_prices:
                    if not (mp.value is None or mp.currency is None):
                        new_mp.append(mp)
                if subscription.min_prices != new_mp:
                    changes = True
                    log.info('Fix subscription %s min_price', qkey)
                    subscription.min_prices = new_mp

                for filter_mp_bundle in subscription.filtered_minprices:
                    new_mp = []
                    for mp in filter_mp_bundle.min_prices:
                        if not (mp.value is None or mp.currency is None):
                            new_mp.append(mp)
                    if filter_mp_bundle.min_prices != new_mp:
                        changes = True
                        log.info(
                            'Fix subscription %s min_price for filter %s',
                            qkey,
                            filter_mp_bundle.filter,
                        )
                        filter_mp_bundle.min_prices = new_mp
                if changes:
                    log.info('Saved fixed subscription %s with minprice', qkey)
                    subscription.save()
            except AttributeError:
                invalid_subscriptions.add(subscription.qkey)

        prepared += 1
        if float(prepared) / float(total) > showed:
            log.info('%.0f%%', showed * 100)
            showed += .1
    log.info('Done preparations. Found %d invalid subscriptions', len(invalid_subscriptions))

    for qkey in invalid_subscriptions:
        try:
            qkey_struct = structure_from_qkey(qkey)
            if qkey_struct.date_forward.date() < today:
                log.info('Deleting old invalid subscription %s', qkey)
                Subscription.objects(qkey=qkey).delete()
            else:
                log.info('Leaving relevant invalid subsription %s', qkey)
        except:
            log.exception('Error managing invalid_subscription %s', qkey)

    log.info('Revalidating subscriptions. This can take a while...')
    for subscription in iterate_chunks(Subscription, 'qkey', Subscription.objects.all().no_cache()):
        try:
            subscription.validate()
        except:
            log.error('Unmanaged invalid subscription %s', subscription.qkey)

    log.info('Done cleanup')
