# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from travel.avia.avia_api.avia.v1.email_dispenser.helpers.currency import currency_to_symbol
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.mysql_cache import (
    avia_data_currency_cache, get_company_avatar,
)
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.variants_filter import get_datetime_with_offset
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.yt_cache import Company
from travel.avia.avia_api.avia.v1.model.subscriber import FRONTEND_ORDER_DT_FORMAT

EMAIL_DATE_FORMAT = '%d.%m'

EMAIL_TIME_FORMAT = '%H:%M'

ru_months = (
    u'января', u'февраля', u'марта', u'апреля', u'мая', u'июня',
    u'июля', u'августа', u'сентября', u'октября', u'ноября', u'декабря'
)
MONTHS = dict(enumerate(ru_months, start=1))


def _format_date(dt):
    return u'{}&nbsp;{}'.format(dt.day, MONTHS[dt.month])


class LetterBlockBuilder(object):
    def __init__(self, yt, logger):
        """

        :param yt.YtClient yt:
        :param logging.logger logger:
        """
        self.yt = yt
        self.logger = logger
        self.company = Company(self.yt)

    def build_for_subscription(self, subscription, variants, applied_filter=None, price_difference=None):
        if not variants:
            return None
        minprice_section_variant = variants.get('min_price_variant_data')
        if not minprice_section_variant:
            self.logger.info('No min price variants passed for subscription %r', subscription)
            return None

        minprice_section_variant_data = list(
            self.variants_data(
                subscription=subscription,
                variants=minprice_section_variant,
                applied_filter=applied_filter,
                price_difference=price_difference,
            )
        )

        popular_section_variants = variants.get('popular_section_variants_data')
        popular_section_variants_data = None
        if popular_section_variants:
            popular_section_variants_data = list(
                self.variants_data(
                    subscription=subscription,
                    variants=popular_section_variants,
                    applied_filter=applied_filter,
                    price_difference=price_difference,
                )
            )
        others = variants.get('other_data')
        other_data = None
        if others:
            other_data = list(
                self.variants_data(
                    subscription=subscription,
                    variants=others,
                    applied_filter=applied_filter,
                    price_difference=price_difference,
                )
            )
        return {
            'min_price_variant_data': minprice_section_variant_data,
            'popular_section_variants_data': popular_section_variants_data,
            'other_data': other_data,
        }

    def variants_data(self, subscription, variants, applied_filter=None, price_difference=None):
        for variant in variants:
            try:
                forward_data = self.segments_data(variant['forward_segments'])
                backward_data = None
                if variant['backward_segments']:
                    backward_data = self.segments_data(variant['backward_segments'])

                order_link = subscription.order_link(
                    # TODO: comparing variant's price with previous min price might be a better option
                    forward_flights=route_from_segments_list(variant['forward_segments']),
                    backward_flights=route_from_segments_list(variant['backward_segments']),
                    filter_fragment=applied_filter.frontend_filter_postfix if applied_filter else None,
                )
                currency_data = avia_data_currency_cache.by_id.get(variant['national_currency_id'])
                currency = currency_data.code if currency_data else None

                variant_data = {
                    'forward': forward_data,
                    'backward': backward_data,
                    'price': format_price_value_html(variant['national_price'] / 100.0),
                    'currency': currency_to_symbol(currency),
                    'order_link': order_link,
                }
                yield variant_data
            except Exception:
                self.logger.exception(
                    'Error building varuants data for qkey %s variant %s applied_filter %s price_difference %s',
                    subscription.qkey,
                    variant,
                    applied_filter,
                    price_difference,
                )

    def segments_data(self, segments):
        if not segments:
            return {}
        data = {
            'departure_time': 'departure_default_value',
            'arrival_time': 'arrival_default_value',
            'departure_date': 'deprature_default_date',
            'arrival_date': 'arrival_default_date',
            'airlines': 'airlines_default_value',
            'airlines_count': 0,
            'airline_avatar': 'airlines_avatar_default_url',
            'connections': 'Без пересадок_default',
        }
        companies_id = {segment['company_id'] for segment in segments}
        companies_title = {
            self.company.by_id[company]['title']
            for company in companies_id
        }
        departure_time = get_datetime_with_offset(segments[0]['departure_time'], segments[0]['departure_offset'])
        arrival_time = get_datetime_with_offset(segments[-1]['arrival_time'], segments[-1]['arrival_offset'])
        data['departure_time'] = departure_time.strftime(EMAIL_TIME_FORMAT)
        data['departure_date'] = departure_time.strftime(EMAIL_DATE_FORMAT)
        data['arrival_time'] = arrival_time.strftime(EMAIL_TIME_FORMAT)
        data['arrival_date'] = arrival_time.strftime(EMAIL_DATE_FORMAT)

        data['airlines'] = ', '.join(companies_title)
        data['airlines_count'] = len(companies_id)
        airlines_avatars = get_airlines_avatars_url(companies_id)
        if len(airlines_avatars) == 1:
            data['airline_avatar'] = airlines_avatars[0]
        else:
            data['airline_avatar'] = None

        data['connections'] = get_connections_string(segments)
        return data


def get_airlines_avatars_url(airlines_set):
    """
    :param set[int] airlines_set:
    :return:
    :rtype: list[basestring]
    """
    assert isinstance(airlines_set, set)
    return filter(None, (
        get_company_avatar(company_id)
        for company_id in airlines_set
    ))


def get_connections_string(segments, lang='ru'):
    connections_template = None
    if len(segments) == 1:
        connections_template = 'Без&nbsp;пересадок'
    elif len(segments) == 2:
        connections_template = '{:d}&nbsp;пересадка'
    elif 3 <= len(segments) <= 5:
        connections_template = '{:d}&nbsp;пересадки'
    elif 6 <= len(segments) <= 21:
        connections_template = '{:d}&nbsp;пересадок'
    return connections_template.format(len(segments) - 1)


def route_from_segments_list(segments_list):
    if not segments_list:
        return None
    string_list = []
    for segment in segments_list:
        string_list.append(
            '.'.join([
                segment['route'],
                (
                    datetime.utcfromtimestamp(segment['departure_time']) +
                    timedelta(seconds=segment['departure_offset'])
                ).strftime(FRONTEND_ORDER_DT_FORMAT),
            ])
        )
    return ','.join(string_list)


def format_price_value_html(value, lang='ru'):
    return '{:,}'.format(int(round(value))).replace(',', '&#8239;')


def format_price_diff_value_html(diff, lang='ru'):
    diff_str = format_price_value_html(abs(diff))
    if diff > 0:
        return '+&nbsp;' + diff_str
    return '-&nbsp;' + diff_str
