# -*- coding: utf-8 -*-
from datetime import datetime, date
from collections import namedtuple

QKEY_DATE_FORMAT = '%Y-%m-%d'

PopularityParams = namedtuple('PopularityParams', 'station_from station_to weekday nv')

QkeyStructure = namedtuple(
    'QkeyStructure', 'point_from_key point_to_key date_forward date_backward klass adults children infants nv'
)


def qkey_to_params_tuple(qkey, stations_cache):
    """
    :param basestring qkey: ключ вида s9600215_c2_2019-01-24_None_economy_1_0_0_ru
    :param dict[basestring, dict] stations_cache: словарик со станциями. ключ - код станции (s1234556),
    значение - словарик, одним из ключей которого будет city_id в формате c213
    :rtype: tuple[PopularityParams, PopularityParams | None]
    """
    point_from, point_to, date_forward, date_backward, klass, adults, children, infants, nv = qkey.split('_')
    date_forward = datetime.strptime(date_forward, QKEY_DATE_FORMAT)
    date_backward = datetime.strptime(date_backward, QKEY_DATE_FORMAT) if date_backward != 'None' else None

    station_from = stations_cache.get(point_from, {}).get('city_id') if point_from.startswith('s') else point_from
    station_from = station_from or point_from

    station_to = stations_cache.get(point_to, {}).get('city_id') if point_to.startswith('s') else point_to
    station_to = station_to or point_to

    forward_param = PopularityParams(station_from, station_to, date_forward.weekday(), nv)
    backward_param = None
    if date_backward:
        backward_param = PopularityParams(station_to, station_from, date_backward.weekday(), nv)
    return forward_param, backward_param


def qkey_from_params(
    point_from_key=None, point_to_key=None,
    date_forward=None, date_backward=None,
    klass=None, adults=1, children=0, infants=0,
    nv=None
):
    """
    :param basestring point_from_key: example: "c213", "s65646"
    :param basestring point_to_key: example: "c213", "s65646"
    :param datetime | date | basestring date_forward: example: "2019-06-30", date(2019,6,30), datetime(2019,6,30)
    :param datetime | date | basestring date_backward: same format as date_forward
    :param basestring klass: example: "economy", "business"
    :param basestring | int adults:
    :param basestring | int children:
    :param basestring | int infants:
    :param basestring nv: example: "ru"
    :rtype: basestring
    """
    if not point_from_key:
        raise ValueError('Empty point_from_key')
    if not point_to_key:
        raise ValueError('Empty point_to_key')
    if not date_forward:
        raise ValueError('Empty date_forward')
    if not klass:
        raise ValueError('Empty klass')
    if not nv:
        raise ValueError('Empty nv')

    if isinstance(date_forward, (datetime, date)):
        date_forward = date_forward.strftime(QKEY_DATE_FORMAT)
    elif isinstance(date_forward, basestring):
        date_forward_reformat = datetime.strptime(date_forward, QKEY_DATE_FORMAT)
        date_forward = date_forward_reformat.strftime(QKEY_DATE_FORMAT)
    else:
        raise ValueError('Invalid date_forward. Value %s, type %s' % (date_forward, type(date_forward)))

    if isinstance(date_backward, (datetime, date)):
        date_backward = date_backward.strftime(QKEY_DATE_FORMAT)
    elif isinstance(date_backward, basestring):
        date_backward_reformat = datetime.strptime(date_backward, QKEY_DATE_FORMAT)
        date_backward = date_backward_reformat.strftime(QKEY_DATE_FORMAT)
    elif date_backward is None:
        date_backward = str(date_backward)
    else:
        raise ValueError('Invalid date_backward. Value %s, type %s' % (date_backward, type(date_backward)))

    if isinstance(adults, int):
        adults = str(adults)
    if isinstance(children, int):
        children = str(children)
    if isinstance(infants, int):
        infants = str(infants)
    values = [point_from_key, point_to_key, date_forward, date_backward, klass, adults, children, infants, nv]
    type_ok = all(map(lambda v: isinstance(v, basestring), values))
    if not type_ok:
        raise ValueError('Incorrect input value type')
    return '_'.join(values)


def structure_from_qkey(qkey):
    """
    :param basestring qkey:
    :rtype: QkeyStructure
    """
    point_from, point_to, date_forward, date_backward, klass, adults, children, infants, nv = qkey.split('_')
    date_forward = datetime.strptime(date_forward, QKEY_DATE_FORMAT)
    date_backward = datetime.strptime(date_backward, QKEY_DATE_FORMAT) if date_backward != 'None' else None

    adults = int(adults)
    children = int(children)
    infants = int(infants)

    if adults + children + infants < 1:
        raise ValueError('Incorrect total number of passengers')

    return QkeyStructure(
        point_from_key=point_from,
        point_to_key=point_to,
        date_forward=date_forward,
        date_backward=date_backward,
        klass=klass,
        adults=adults,
        children=children,
        infants=infants,
        nv=nv,
    )
