# -*- coding: utf-8 -*-
import time
from yt.wrapper import TablePath


def filter_cold_qkey(qkeys, ytc, log):
    """
    Отфильтровываем набор qkeys так, что в нём остаются только те направления,
    по которым не было запроса за последние 24 часа

    :param set[str] qkeys: list of active qkeys
    :param ytc: yt client
    :param logging.Logger log:
    :return: list of qkeys that were not preheated
    """
    left_timestamp = time.time() - 24 * 60 * 60
    cold_qkeys = set(qkeys)
    source_table_names = list(ytc.search(
        '//home/logfeller/logs/avia-ticket-daemon-timeline-log/30min',
        node_type='table',
    ))

    log.info('Getting data from %s', source_table_names)
    source_tables = [
        TablePath(table_name, columns=['qid', 'timestamp'])
        for table_name in source_table_names
    ]

    def qkey_mapper(record):
        qkey = record['qid'].split('.')[3]
        unixtime = record['timestamp']
        if unixtime >= left_timestamp and qkey in qkeys:
            yield {'qkey': qkey}

    def reduce_identity(key, values):
        for value in values:
            yield value
            return

    with ytc.TempTable() as temp_table:
        log.info('Created temp table %s', temp_table)
        log.info('Running map-reduce to get distinct qkeys')
        ytc.run_map_reduce(
            source_table=source_tables,
            destination_table=temp_table,
            mapper=qkey_mapper,
            reducer=reduce_identity,
            reduce_by=['qkey']
        )
        log.info('Reading temp table %s to get qkeys', temp_table)
        return filter_out(ytc, temp_table, cold_qkeys, log)


def filter_out(ytc, hot_qkeys_table, cold_qkeys, log):
    hot_qkeys = ytc.read_table(hot_qkeys_table)
    log.info('There are %d qkeys passed to filtering function', len(cold_qkeys))
    for requested in hot_qkeys:
        if requested['qkey'] in cold_qkeys:
            cold_qkeys.remove(requested['qkey'])
    log.info('There are %d uncovered qkeys, preheat them!', len(cold_qkeys))
    return cold_qkeys
