# -*- coding: utf-8 -*-
from datetime import datetime, timedelta
from itertools import islice

from travel.avia.avia_api.avia.v1.email_dispenser.helpers.qkey_utils import qkey_to_params_tuple
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.mysql_cache import get_popularity_map
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.yt_cache import Station
from travel.avia.avia_api.avia.v1.email_dispenser.variants_fetcher.variants_log_yt import (
    VariantsLogSubscriptionPipeline,
)
from travel.avia.avia_api.avia.v1.model.variants_compute_history import VariantsYtHistory, SubscriptionPreheatHistory

TEMP_YT_DIR = '//home/avia/tmp/'


# YT Pipeline runner

def fetch_subscription_variants_info(ytc, qkeys, source_tables, left_date, right_date, stations_cache, logger):
    """
    :param ytc: yt wrapper
    :param set[basestring] qkeys:
    :param list[basestring] source_tables: list of source tables to get variants from
    :param datetime.datetime left_date:
    :param datetime.datetime right_date:
    :param dict[basestring, dict] stations_cache:
    :param logging.Logger logger:
    :return: Result table paths (min_prices_variants, min_prices, top_popular)
    :rtype: tuple[basestring, basestring, basestring]
    """
    qkey_params_set = set()

    for qkey in qkeys:
        for tup in qkey_to_params_tuple(qkey, stations_cache):
            if tup is not None:
                qkey_params_set.add(tup)

    logger.info('Parsed qkeys for popularity map. %d records total', len(qkey_params_set))

    popularity = get_popularity_map(qkey_params_set)
    logger.info('Got popularirty map. %d records total', len(popularity))
    logger.info('Running map-reduces')

    min_prices_variants = ytc.create_temp_table(prefix=TEMP_YT_DIR)
    min_prices = ytc.create_temp_table(prefix=TEMP_YT_DIR)
    top_popular = ytc.create_temp_table(prefix=TEMP_YT_DIR)

    yt_pipeline = VariantsLogSubscriptionPipeline(ytc, qkeys, left_date, right_date, logger)
    yt_pipeline.run_pipeline(
        source_tables, min_prices_variants, min_prices,
        top_popular, popularity,
        stations_cache
    )

    logger.info('See minimal price variants here %s', min_prices_variants)
    logger.info('See min prices for each qkey here: %s', min_prices)
    logger.info('See top popular variants %s', top_popular)
    return min_prices_variants, min_prices, top_popular


def ensure_recent_computations(ytc, subscriptions, logger):
    """
    :param ytc: yt wrapper
    :param typing.Iterable[avia.v1.model.subscriber.Subscription] subscriptions:
    :param logging.Logger logger:
    :rtype: VariantsYtHistory
    """
    right_date_time = datetime.utcnow()
    left_date_time = datetime.utcnow() - timedelta(days=1)

    recent_computation = VariantsYtHistory.recent()

    recent_heater = SubscriptionPreheatHistory.recent()
    have_new_preheated_data = False
    if recent_heater and recent_computation:
        time_passed_since_last_preheat = datetime.utcnow() - timedelta(hours=2) > recent_heater.preheated_at
        variants_computed_before_preheat = recent_heater and recent_computation.computed_at < recent_heater.preheated_at
        have_new_preheated_data = time_passed_since_last_preheat and variants_computed_before_preheat

    if not recent_computation or have_new_preheated_data:
        logger.info('No recent computations found. Computing a new set')
        qkeys = {s.qkey for s in subscriptions}
        # Dangerous. Depends on timezone avia-variants-log is written
        source_tables = list(ytc.search('//home/logfeller/logs/avia-variants-log/30min', node_type='table'))
        logger.info(
            'Going to get variants from %s for %s%s',
            source_tables,
            list(islice(qkeys, 10)),
            (' + ' + str(len(qkeys) - 10) + ' more') if len(qkeys) > 10 else '',
        )
        logger.info('Caching station from yt')
        stations_cache = Station(ytc).by_id
        logger.info('Cached %d stations', len(stations_cache))
        min_prices_variants, min_prices, top_popular = fetch_subscription_variants_info(
            ytc, qkeys, source_tables, left_date_time, right_date_time, stations_cache, logger
        )
        VariantsYtHistory(
            min_prices_variants_table=min_prices_variants,
            min_prices_qkey_table=min_prices,
            popular_variants_table=top_popular,
        ).save()
        logger.info('Variants are reduced, now we can generate letters for people.')
    recent_computation = VariantsYtHistory.recent()
    if not recent_computation:
        raise Exception(
            'Could not load recent computation table names, '
            'although they were recently generated'
        )
    return recent_computation
