# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from datetime import datetime, timedelta

from flask import request
from marshmallow import post_load

import travel.avia.avia_api.ant
import travel.avia.avia_api.ant.api_interface
import travel.avia.avia_api.ant.custom_types
import travel.avia.avia_api.ant.exceptions
from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.ant.custom_types import ArgUUID, ArgPassportId
from travel.avia.avia_api.avia.commands import notify_user_if_min_price_has_changed
from travel.avia.avia_api.avia.lib.decorators import refuse_None
from travel.avia.avia_api.avia.lib.jsend import ApiFail
from travel.avia.avia_api.avia.lib.passport_utils import get_yandex_uid_by_oauth_token
from travel.avia.avia_api.avia.lib.remote_addr import real_client_ip
from travel.avia.avia_api.avia.lib.user_flights import add_flight_manually, UnknownFlightException
from travel.avia.avia_api.avia.v1.internal.api import internal_api
from travel.avia.avia_api.avia.v1.model.device import Device
from travel.avia.avia_api.avia.v1.model.favorite import Price
from travel.avia.avia_api.avia.v1.model.flight import Flight
from travel.avia.avia_api.avia.v1.model.flight_status import (
    FlightStatusCancel, FlightStatusActual, FlightStatusGate,
    FlightUpdateStatusError, FlightIsAlreadyCancelledError
)
from travel.avia.avia_api.avia.v1.model.user import User
from travel.avia.avia_api.avia.v1.schemas import (
    fields, ApiSchema, TariffSchema,
    FlightSchema, FlightIdSchema, FlightListViewSchema,
)
from travel.avia.avia_api.avia.v1.views.favorites import FavoriteSchema

log = logging.getLogger(__name__)


class UpdatedFlightSchema(FlightSchema):
    gate = fields.Integer(required=False)


class FlightGetterSchema(ApiSchema):
    number = fields.String(required=True)
    departure_date = fields.DateTime(format='%Y-%m-%d', required=True)
    departure = fields.String(required=False)
    arrival = fields.String(required=False)

    @post_load
    def make_object(self, data):
        log.info('FlightGetterSchema: %r', data)
        departure_date = data['departure_date']
        try:
            return Flight.objects.get(
                number=data['number'],
                departure_date=datetime(
                    departure_date.year,
                    departure_date.month,
                    departure_date.day
                ),
                **{
                    key: data[key]
                    for key in ('arrival', 'departure')
                    if key in data
                }
            )
        except Flight.DoesNotExist:
            raise ApiFail('Flight not found')
        except Flight.MultipleObjectsReturned:
            raise ApiFail('Ambiguous query: multiple flights are found')


@internal_api.view('/flight/list/raw/')
def flight_list_raw():
    return Flight.objects


@internal_api.view('/flight/list/update_gate/')
@internal_api.result_schema(FlightIdSchema, many=True)
def flight_list_update_gate(schema_context=None):

    """ Получить набор рейсов для обновления гейтов """

    return Flight.objects


@internal_api.view('/flight/list/update_from_tablo/')
@internal_api.result_schema(FlightIdSchema, many=True)
def flight_list_update_from_tablo(schema_context=None):

    """ Получить набор рейсов для обновления статусов """

    # Обновлять статусы для рейсов на неделю вперёд
    right_border = datetime.now() + timedelta(days=7)
    left_border = datetime.now() - timedelta(days=2)

    return Flight.objects.filter(
        departure_date__gte=left_border,
        departure_date__lte=right_border
    )


@internal_api.view('/flight/update/cancel/',
                   disable_auth_in_environments=['testing'])
@internal_api.result_schema(FlightSchema)
def flight_update_cancel(
    flight=ViewParam(schema=FlightGetterSchema, about='Рейс'),
    schema_context=None
):
    """ Отмена рейса """

    try:
        flight.update_status(FlightStatusCancel())
    except FlightIsAlreadyCancelledError:
        log.info('Flight %r is already cancelled', flight)
    except FlightUpdateStatusError as exc:
        log.exception('Flight cancelling error: %s', exc)
        flight.reload()
    else:
        flight.save()
        log.info('Flight %r is cancelled successfully', flight)

    return flight


DT_FMT = '%Y-%m-%d %H:%M'
DT_FMT_LEN = len(datetime.now().strftime(DT_FMT))


@refuse_None
def parse_dthm(text):
    normalized = ' '.join(text.strip().split())
    dt = datetime.strptime(normalized[:DT_FMT_LEN], DT_FMT)

    return datetime(dt.year, dt.month, dt.day, dt.hour, dt.minute)


class ArgDateHM(travel.avia.avia_api.ant.custom_types.ArgType):
    fmt = 'YYYY-MM-DD hh:mm'

    def clean(self, raw):
        try:
            return parse_dthm(raw)
        except Exception:
            log.exception('Wrong datetime format: %r', raw)
            raise travel.avia.avia_api.ant.exceptions.ValidationError('Wrong datetime format: %r' % raw)


@internal_api.view('/flight/update/', disable_auth_in_environments=['testing'])
@internal_api.result_schema(UpdatedFlightSchema)
def flight_update(
    flight=ViewParam(schema=FlightGetterSchema, about='Рейс'),
    actual_datetime=ViewParam(type_=ArgDateHM(), required=False),
    gate=ViewParam(required=False),
    schema_context=None
):
    """ Обновление данных рейса """

    log.info('Flight update: %r: actual_datetime: %r, gate: %r',
             flight, actual_datetime, gate)

    if actual_datetime:
        try:
            flight.update_status(
                FlightStatusActual(actual_datetime=actual_datetime)
            )
        except FlightIsAlreadyCancelledError:
            pass
        except FlightUpdateStatusError as exc:
            log.warning(
                'Flight %r actual datetime update error: %r', flight, exc
            )
        else:
            flight.save()

    if gate:
        try:
            flight.update_status(
                FlightStatusGate(gate=gate)
            )
        except FlightIsAlreadyCancelledError:
            pass
        except FlightUpdateStatusError as exc:
            log.exception('Flight %r gate update error: %r', flight, exc)
        else:
            flight.save()

    return flight.update_upsert_reload()


@internal_api.view('/favorite/update/', disable_auth_in_environments=['testing'])
@internal_api.result_schema(FavoriteSchema)
def favorite_update(
    uuid=ViewParam(type_=ArgUUID(), required=False),
    oauth_token=ViewParam(required=False),
    key=ViewParam(required=True),
    min_price=ViewParam(schema=TariffSchema, required=True),
    schema_context=None,
):
    """ Обновление данных рейса """

    log.info(
        'Favorite update: uuid: %r, oauth_token: %r, key: %r, min_price: %r',
        uuid, oauth_token, key, min_price
    )

    user = None

    if uuid:
        users = list(User.objects(uuid=uuid))
        log.info('By users: uuid %r -> %r', uuid, users)
        if len(users) == 1:
            [user] = users
        else:
            users = {d.user for d in Device.objects(uuid=uuid)}
            log.info('By devices: uuid %r -> %r', uuid, users)
            if len(users) == 1:
                [user] = users

    if user is None and oauth_token:
        yandex_uid = get_yandex_uid_by_oauth_token(
            oauth_token,
            userip=real_client_ip(request),
        )
        log.info('%r -> %r', oauth_token, yandex_uid)

        if yandex_uid:
            users = list(User.objects(yandex_uid=yandex_uid))
            log.info('By yandex_uid: %r -> %r', yandex_uid, users)
            if len(users) == 1:
                [user] = users

    if user is None:
        log.info('User by %r and %r is not found', uuid, oauth_token)
        raise ApiFail('User is not found')

    favorites = [fave for fave in user.favorites if fave.key == key]

    if len(favorites) != 1:
        available_keys = [fave.key for fave in user.favorites]
        log.info(
            "Favorite with key %r isn't found. Available keys: %r",
            key, available_keys
        )
        raise ApiFail(
            {'available_keys': available_keys}
        )

    [favorite] = favorites

    favorite.min_price = Price(
        value=min_price['value'],
        currency=min_price['currency'],
    )

    try:
        notify_user_if_min_price_has_changed(user, favorite)
        user.save()
    except Exception as exc:
        log.exception(
            "Couldn't notify user: %r. User fields: %r",
            exc, user.to_json()
        )

    user.reload()
    [favorite] = [fave for fave in user.favorites if fave.key == key]

    return favorite


@internal_api.view(
    '/flight/add/',
    methods=['POST'],
    disable_auth_in_environments=['development'],
)
@internal_api.result_schema(FlightListViewSchema)
def flight_add(
    passport_id=ViewParam(type_=ArgPassportId(), required=True),
    flight_number=ViewParam(required=True),
    departure_date=ViewParam(type_=travel.avia.avia_api.ant.custom_types.Date(), required=True),
    schema_context=None,
):
    User.objects(
        yandex_uid=passport_id
    ).update(
        set__touched=datetime.utcnow(),
        upsert=True,
    )

    user = User.objects.get(yandex_uid=passport_id)

    try:
        return add_flight_manually(user, flight_number, departure_date)
    except UnknownFlightException as exc:
        raise ApiFail({
            'unknown_flights': [{
                'number': exc.number,
                'departure_datetime': exc.departure_date.strftime('%Y-%m-%d %H:%M')
            }]
        })


@internal_api.view(
    '/flight/drop/',
    methods=['POST'],
    disable_auth_in_environments=['development'],
)
@internal_api.result_schema(FlightListViewSchema)
def flight_drop(
    passport_id=ViewParam(type_=ArgPassportId(), required=True),
    key=ViewParam(required=True),
    schema_context=None,
):
    user = User.objects(
        yandex_uid=passport_id
    )

    if user:
        user.update(pull__aeroex_notify_flights=Flight(id=key))
        user.update(pull__flights=Flight(id=key))

    return 'ok'
