import logging
import uuid
from datetime import datetime

from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.geo import Point
from travel.avia.library.python.common.xgettext import i18n

from travel.avia.avia_api.avia.lib.min_price_storage import (
    MinPriceStorageChain, MinPriceMemcachedStorage, MinPriceMysqlStorage,
    FlightType
)
from travel.avia.avia_api.avia.v1.model.db import db
from travel.avia.avia_api.avia.v1.model.passengers import Passengers

log = logging.getLogger(__name__)


class Price(db.EmbeddedDocument):
    value = db.FloatField(required=True)
    currency = db.StringField(max_length=3, required=True)

    def __str__(self):
        currency = Currency.objects.get(code=self.currency)

        return i18n.stringify(i18n.xformat(
            currency.format_value(self.value, show_cents=False),
            whole=lambda w: w,
            unit=lambda j: j
        ))


def str_uuid4():
    return str(uuid.uuid4())


class MinPriceNotification(db.EmbeddedDocument):
    price = db.EmbeddedDocumentField(Price, required=True)
    when = db.DateTimeField(required=False)


class Favorite(db.EmbeddedDocument):
    key = db.StringField(required=True, default=str_uuid4)

    point_from_key = db.StringField(required=True)
    point_to_key = db.StringField(required=True)

    date_forward = db.DateTimeField(required=True)
    date_backward = db.DateTimeField()

    direct_only = db.BooleanField(default=False)

    passengers = db.EmbeddedDocumentField(Passengers)
    service = db.StringField(required=True)
    variants = db.IntField(required=True)
    min_price = db.EmbeddedDocumentField(Price, required=True)

    last_min_price_notification = db.EmbeddedDocumentField(
        MinPriceNotification, required=True
    )

    filter_params = db.DictField(default=dict)

    @staticmethod
    def get_point_by_key(key):
        try:
            return Point.get_any_by_key(key)
        except Exception:
            log.warning('No point with key [%r]', key, exc_info=True)
            return None

    @property
    def point_from(self):
        return self.get_point_by_key(self.point_from_key)

    @property
    def point_to(self):
        return self.get_point_by_key(self.point_to_key)

    def is_relevant(self):
        # TODO: localize date_forward and check the relevance more precisely
        return self.date_forward.date() >= datetime.utcnow().date()

    def update_min_price(self, user):
        if not self.point_to:
            log.info('No point_to of User(%s) favorite(%s)', user, self)
            return

        if not self.point_from:
            log.info('No point_from of User(%s) favorite(%s)', user, self)
            return

        storage_options = dict(
            point_from=self.point_from,
            point_to=self.point_to,
            departure_date=self.date_forward,
            return_date=self.date_backward,
            national_version='ru',
            currency_code=self.min_price.currency,
        )

        storage_chain = MinPriceStorageChain(
            MinPriceMemcachedStorage(**storage_options),
            MinPriceMysqlStorage(**storage_options),
        )

        direct_price = storage_chain.get_min_price(
            FlightType.Direct, self.passengers
        )

        indirect_price = storage_chain.get_min_price(
            FlightType.Indirect, self.passengers
        ) if not self.direct_only else None

        min_prices = filter(None, [direct_price, indirect_price])

        if not min_prices:
            log.info(
                'No min prices for favorite [%s] of user [%s]', self, user
            )
            return None

        stored_min_price = min(min_prices, key=lambda price: price.value)

        self.min_price.value = stored_min_price.value

    def __str__(self):
        return '[%s-%s] (%s-%s) %s [%s] [%s]vs' % (
            self.point_from_key,
            self.point_to_key,
            self.date_forward,
            self.date_backward,
            self.passengers.format_key(),
            self.service,
            self.variants,
        )

    def __repr__(self):
        return '<%s:%s>' % (self.__class__.__name__, self.key)
