# -*- coding: utf-8 -*-
import datetime

from enum import Enum

from travel.avia.avia_api.avia.v1.model.db import db


# https://github.yandex-team.ru/avia/avia-frontend/blob/release/server/controllers/search.ts

class TransferFilter(db.EmbeddedDocument):
    meta = {
        'strict': False,
    }
    count = db.IntField(required=False, null=True)
    min_duration = db.IntField(required=False, null=True, db_field='minDuration')
    # not supported at frontend
    max_duration = db.IntField(required=False, null=True, db_field='maxDuration')
    # not supported at frontend
    has_airport_change = db.BooleanField(required=False, null=True, db_field='hasAirportChange')
    has_night = db.BooleanField(required=False, null=True, db_field='hasNight')

    def __str__(self):
        return self.to_json()


class TimeOfDay(Enum):
    MORNING = 1
    DAY = 2
    EVENING = 3
    NIGHT = 4


class TimeHelper(object):
    TIME_TO_HOURS_MAP = {
        TimeOfDay.NIGHT: (0, 5),
        TimeOfDay.MORNING: (6, 11),
        TimeOfDay.DAY: (12, 17),
        TimeOfDay.EVENING: (18, 23),
    }

    @classmethod
    def is_time_of_day(cls, dt, timeofday):
        assert isinstance(timeofday, TimeOfDay)
        assert isinstance(dt, datetime.datetime)
        left, right = cls.TIME_TO_HOURS_MAP[timeofday]
        hour = dt.hour
        if left <= hour <= right:
            return True
        return False

    @classmethod
    def is_night_in_range(cls, dt_start, dt_stop):
        """
        Detects if date range from dt_start to dt_stop contains night
        :param datetime.datetime dt_start:
        :param datetime.datetime dt_stop:
        :return:
        """
        if dt_start is None or dt_stop is None:
            return False
        if dt_start > dt_stop:
            dt_start, dt_stop = dt_stop, dt_start
        if dt_start.date() != dt_stop.date():
            return True
        return (
            cls.is_time_of_day(dt_stop, TimeOfDay.NIGHT) or
            cls.is_time_of_day(dt_start, TimeOfDay.NIGHT)
        )


class TimeFilter(db.EmbeddedDocument):
    meta = {
        'strict': False,
    }
    forward_arrival = db.IntField(required=False, null=True, db_field='forwardArrival')
    forward_departure = db.IntField(required=False, null=True, db_field='forwardDeparture')
    backward_arrival = db.IntField(required=False, null=True, db_field='backwardArrival')
    backward_departure = db.IntField(required=False, null=True, db_field='backwardDeparture')

    def __str__(self):
        return self.to_json()


class AirportFilter(db.EmbeddedDocument):
    meta = {
        'strict': False,
    }
    forward_departure = db.ListField(
        db.IntField(required=True, null=False), required=False, null=True,
        db_field='forwardDeparture'
    )
    forward_arrival = db.ListField(
        db.IntField(required=True, null=False), required=False, null=True,
        db_field='forwardArrival'
    )
    forward_transfers = db.ListField(
        db.IntField(required=True, null=False), required=False, null=True,
        db_field='forwardTransfers'
    )
    backward_departure = db.ListField(
        db.IntField(required=True, null=False), required=False, null=True,
        db_field='backwardDeparture'
    )
    backward_arrival = db.ListField(
        db.IntField(required=True, null=False), required=False, null=True,
        db_field='backwardArrival'
    )
    backward_transfers = db.ListField(
        db.IntField(required=True, null=False), required=False, null=True,
        db_field='backwardTransfers'
    )

    def __str__(self):
        return self.to_json()


class Filter(db.EmbeddedDocument):
    meta = {
        'strict': False,
    }
    with_baggage = db.BooleanField(required=False, null=True, db_field='withBaggage')
    airlines = db.ListField(db.IntField(required=True), required=False, null=True, db_field='airlines')
    transfer_filters = db.EmbeddedDocumentField(TransferFilter, required=False, null=True, db_field='transfer')
    time_filters = db.EmbeddedDocumentField(TimeFilter, required=False, null=True, db_field='time')
    airport_filters = db.EmbeddedDocumentField(AirportFilter, required=False, null=True, db_field='airport')
    frontend_filter_postfix = db.StringField(required=False, null=True, db_field='filter_url_postfix')

    def __str__(self):
        return self.to_json()
