# -*- coding: utf-8 -*-
import logging
from datetime import timedelta, datetime

from django.conf import settings

from travel.avia.library.python.common.models.geo import Point
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.route_search.base import PlainSegmentSearch

from travel.avia.avia_api.avia.v1.model.aeroexpress import Aeroexpress
from travel.avia.avia_api.avia.v1.model.db import db
from travel.avia.avia_api.avia.v1.model.flight_status import (
    FlightStatus, FlightStatusCancel, FlightStatusActual, FlightStatusGate,
    FlightUpdateStatusError
)

log = logging.getLogger(__name__)


def find_aeroexpress_segment(airport_from, airport_to, aware_flight_datetime):
    if airport_from.settlement is None:
        log.debug('Skip find_aeroexpress_segment: no airport_from.settlement')
        return None

    if airport_from.id not in settings.AEROEXPRESSES:
        log.debug('Skip find_aeroexpress_segment. airport without ax: %s',
                  airport_from.id)
        return None

    # полтора часа до вылета внутреннего рейса и два с половиной часа до
    # вылета международного рейса
    transfer_gap = timedelta(minutes=(
        150
        if airport_from.get_country_id() != airport_to.get_country_id() else
        90
    ))

    max_aeroexpress_arrival_dt = aware_flight_datetime - transfer_gap
    min_aeroexpress_arrival_dt = (
        max_aeroexpress_arrival_dt - timedelta(minutes=90)
    )

    suburban_segments = PlainSegmentSearch(
        airport_from.settlement, airport_from, 'suburban'
    ).gen_from(max_aeroexpress_arrival_dt - timedelta(days=1))

    aeroexpress_segment = None

    for segment in suburban_segments:
        if not segment.thread.is_aeroexpress:
            continue

        # RASPTICKETS-4260 Ограничить максимальный промежуток времени
        # до ближайшего Аэроэкспресса
        if segment.arrival < min_aeroexpress_arrival_dt:
            continue

        if segment.arrival > max_aeroexpress_arrival_dt:
            break

        aeroexpress_segment = segment

    return aeroexpress_segment


def get_aeroexpress(airport_from, airport_to, flight_datetime):
    aware_flight_datetime = airport_from.pytz.localize(flight_datetime)

    aeroexpress_segment = find_aeroexpress_segment(
        airport_from, airport_to, aware_flight_datetime
    )

    if aeroexpress_segment is not None:
        return Aeroexpress.from_segment(aeroexpress_segment)


class Flight(db.Document):
    number = db.StringField(max_length=8, required=True)

    # Пункты отправления и прибытия
    departure = db.StringField()
    arrival = db.StringField()

    # Время (по расписанию) отправления и прибытия
    departure_date = db.DateTimeField(
        required=True, unique_with=('number', 'departure')
    )
    departure_datetime = db.DateTimeField()
    arrival_datetime = db.DateTimeField()
    path_minutes = db.IntField()

    statuses = db.ListField(db.EmbeddedDocumentField(FlightStatus))

    terminal = db.StringField()
    company_id = db.IntField()

    aeroexpress = db.EmbeddedDocumentField(Aeroexpress)

    touched = db.DateTimeField(default=datetime.utcnow)

    @property
    def key(self):
        return str(self.id)

    @classmethod
    def get_or_create(cls, number, departure, departure_date):
        try:
            return cls.objects.get(
                number=number,
                departure=departure,
                departure_date=departure_date,
            )
        except cls.DoesNotExist:
            new_flight = cls(
                number=number,
                departure=departure,
                departure_date=departure_date,
            )
            new_flight.save()
            return new_flight

    def update_upsert_reload(self):
        # Переписать данные в монге и обновиться
        id_keys = dict(
            number=self.number,
            departure_date=self.departure_date,
            departure=self.departure
        )

        Flight.objects(**id_keys).update_one(
            upsert=True,
            departure_datetime=self.departure_datetime,
            arrival_datetime=self.arrival_datetime,
            statuses=self.statuses,
            terminal=self.terminal,
            company_id=self.company_id,
            path_minutes=self.path_minutes,
            arrival=self.arrival,
            aeroexpress=self.aeroexpress,
            touched=datetime.utcnow()
        )

        return Flight.objects.get(**id_keys)

    @property
    def canceled(self):
        return bool(self.specific_statuses(FlightStatusCancel))

    def specific_statuses(self, status_cls):
        return [s for s in self.statuses if isinstance(s, status_cls)]

    @property
    def real_departure(self):
        try:
            actual_status = self.specific_statuses(FlightStatusActual)[-1]
        except IndexError:
            return

        return actual_status.actual_datetime

    @property
    def any_departure_dt(self):
        return self.real_departure or self.departure_datetime

    @property
    def arrival_datetime_planed(self):
        delayed_by = self.any_departure_dt - self.departure_datetime
        return self.arrival_datetime + delayed_by

    @property
    def delayed(self):
        return self.any_departure_dt != self.departure_datetime

    @property
    def checkin_started(self):
        return datetime.utcnow() + timedelta(days=1) > self.departure_datetime

    @property
    def gate(self):
        try:
            return self.specific_statuses(FlightStatusGate)[-1].gate
        except IndexError:
            return None

    @property
    def company(self):
        if self.company_id is None:
            return None

        try:
            return Company.objects.get(id=self.company_id)
        except Company.DoesNotExist:
            return None
        except Exception as exc:
            log.error(
                "Couldn't find a company by id %r: %r", self.company_id, exc
            )
            return None

    def update_status(self, status):
        try:
            status.apply_to_flight(self)
        except FlightUpdateStatusError as e:
            log.warning('Flight update_status error: %s', e)
            raise

        return status

    def cache_airport_from(self, airport_from=None):
        try:
            if self._airport_from and not airport_from:
                return
        except AttributeError:
            pass

        if airport_from is None:
            airport_from = Point.get_by_key(self.departure)
        elif airport_from.point_key != self.departure:
            raise ValueError("Invalid departure airport (%r != %r)" % (
                airport_from.point_key, self.departure
            ))
        self._airport_from = airport_from

    @property
    def airport_from(self):
        self.cache_airport_from()
        return self._airport_from

    def cache_airport_to(self, airport_to=None):
        try:
            if self._airport_to and not airport_to:
                return
        except AttributeError:
            pass

        if airport_to is None:
            airport_to = Point.get_by_key(self.arrival)
        elif airport_to.point_key != self.arrival:
            raise ValueError("Invalid arrival airport (%r != %r)" % (
                airport_to.point_key, self.arrival
            ))
        self._airport_to = airport_to

    @property
    def airport_to(self):
        self.cache_airport_to()
        return self._airport_to

    def update_aeroexpress(self):
        new_aeroexpress = get_aeroexpress(
            self.airport_from, self.airport_to,
            self.any_departure_dt
        )

        if self.aeroexpress is not None and new_aeroexpress is not None:
            attributes_to_be_equal = [
                'station', 'datetime', 'arrival_datetime',
                'price_standart', 'price_business', 'price_currency'
            ]

            for attr in attributes_to_be_equal:
                if self.aeroexpress[attr] != new_aeroexpress[attr]:
                    log.info(
                        'Aeroexpress for flight %r has been changed: %r -> %r',
                        self, self.aeroexpress.to_json(),
                        new_aeroexpress.to_json()
                    )
                    break
            else:
                # the new aeroexpress overwrites notification_status otherwise
                log.info('New aeroexpress for flight %r is the same', self)
                return

        self.aeroexpress = new_aeroexpress

    def update_path_minutes(self, path_minutes=None):
        if not path_minutes:
            path_minutes = int(
                (self.utc_arrival_dt - self.utc_departure_dt).
                total_seconds() / 60
            )

        self.path_minutes = path_minutes

        log.debug('flight path_minutes: %r', self.path_minutes)

    @property
    def utc_departure_dt(self):
        localized = self.airport_from.localize(loc=self.departure_datetime)

        log.debug(
            'utc_departure_dt '
            'airport_from:%r  departure_datetime:%r  localized:%r',
            self.airport_from.id, self.departure_datetime, localized
        )

        return localized

    @property
    def utc_arrival_dt(self):
        localized = self.airport_to.localize(loc=self.arrival_datetime)

        log.debug(
            'utc_arrival_dt '
            'airport_to:%r  arrival_datetime:%r  localized:%r',
            self.airport_to.id, self.arrival_datetime, localized
        )

        return localized

    def updated_with_info(self, flight_info):
        flight_info.update_flight(self)

        self.update_aeroexpress()
        self.update_path_minutes()
        return self.update_upsert_reload()

    def __unicode__(self):
        return '[%s] <%s>' % (
            self.number, self.departure_datetime
        )
