# -*- coding: utf-8 -*-
import logging
from datetime import datetime

from travel.avia.avia_api.avia.lib.push_notifications import (
    FlightCancellationPushNotification, DepartureTimeChangePushNotification,
    GateChangePushNotification, CheckinStartPushNotification
)
from travel.avia.avia_api.avia.v1.model.db import db

log = logging.getLogger(__name__)


class FlightUpdateStatusError(Exception):
    pass


class FlightIsAlreadyCancelledError(FlightUpdateStatusError):
    pass


class FlightStatus(db.EmbeddedDocument):
    timestamp = db.DateTimeField(default=datetime.utcnow)
    code = db.StringField()

    meta = {'allow_inheritance': True}

    def apply_to_flight(self, flight):
        # Не добавлять статусы в отменённый рейс
        if flight.canceled:
            raise FlightIsAlreadyCancelledError('Already canceled')

        flight.statuses.append(self)

    def __unicode__(self):
        return 'timestamp[%s] %s' % (self.timestamp, self.code)


class FlightStatusCancel(FlightStatus):
    code = db.StringField(default='cancel')

    def apply_to_flight(self, flight):
        super(FlightStatusCancel, self).apply_to_flight(flight)
        FlightCancellationPushNotification(flight).send()


class FlightStatusActual(FlightStatus):
    code = db.StringField(default='actual_datetime')
    actual_datetime = db.DateTimeField()

    def apply_to_flight(self, flight):
        if flight.real_departure == self.actual_datetime:
            log.info('Actual datetime for %r is not changed', flight)
            return

        prev_departure_dt = flight.real_departure or flight.departure_datetime
        super(FlightStatusActual, self).apply_to_flight(flight)

        if flight.real_departure != prev_departure_dt:
            flight.update_aeroexpress()

            DepartureTimeChangePushNotification(flight).send()

    def __unicode__(self):
        return '%s actual[%s]' % (
            super(FlightStatusActual, self).__unicode__(),
            self.actual_datetime
        )


class FlightStatusGate(FlightStatus):
    code = db.StringField(default='gate')
    gate = db.StringField()

    def apply_to_flight(self, flight):
        super(FlightStatusGate, self).apply_to_flight(flight)
        GateChangePushNotification(flight).send()


class FlightStatusCheckinStarted(FlightStatus):
    code = db.StringField(default='checkin_started')

    def apply_to_flight(self, flight):
        if flight.specific_statuses(FlightStatusCheckinStarted):
            raise FlightUpdateStatusError(
                'Check-in start status has been already applied to the flight'
            )

        super(FlightStatusCheckinStarted, self).apply_to_flight(flight)
        CheckinStartPushNotification(flight).send()
