# -*- coding: utf-8 -*-
import hashlib
import logging
import uuid
from datetime import date, datetime, timedelta
from operator import itemgetter
from urllib import urlencode
from urlparse import urljoin

from django.conf import settings

from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.geo import Point
from travel.avia.library.python.common.utils.iterrecipes import group_by
from travel.avia.library.python.common.xgettext import i18n

from travel.avia.avia_api.avia.lib.min_price_storage import (
    FlightType, MinPriceMemcachedStorage, MinPriceMysqlStorage, MinPriceStorageChain,
    MinPriceYtStorage,
)
from travel.avia.avia_api.avia.lib.feature_flag import use_travel_portal_links
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.qkey_utils import qkey_from_params, structure_from_qkey
from travel.avia.avia_api.avia.v1.model.db import db
from travel.avia.avia_api.avia.v1.model.filters import Filter
from travel.avia.avia_api.avia.v1.model.passengers import Passengers

log = logging.getLogger(__name__)

FORMAT_DATE = '%Y-%m-%dT%H:%M:%S'
FRONTEND_DATE_FMT = '%Y-%m-%d'
FRONTEND_DT_FMT = '%Y-%m-%dT%H:%M:%S'
FRONTEND_ORDER_DT_FORMAT = '%Y-%m-%dT%H:%M'

APPROVED_SESSION_DECAY_TIME = timedelta(weeks=1)
UNAPPROVED_SESSION_DECAY_TIME = timedelta(weeks=1)
ALLOWED_UNAPPROVED_SESSIONS = 3
APPROVED_PASSPORT_DECAY_TIME = timedelta(weeks=100)
UNAPPROVED_PASSPORT_DECAY_TIME = timedelta(weeks=1)


def frontend_host(nv):
    return settings.FRONTEND_URLS.get(nv, settings.FRONTEND_URLS['ru'])


def travel_frontend_host(nv):
    if settings.YANDEX_ENVIRONMENT_TYPE == 'testing':
        return settings.TRAVEL_TEST_FRONTEND_URLS.get(nv, settings.TRAVEL_TEST_FRONTEND_URLS['ru'])
    return settings.TRAVEL_FRONTEND_URLS.get(nv, settings.TRAVEL_FRONTEND_URLS['ru'])


def currency(nv):
    currencies = settings.AVIA_NATIONAL_CURRENCIES
    return currencies.get(nv, currencies['ru'])


def hashed(val):
    return hashlib.sha256(str(val)).hexdigest() if val else None


class MinPrice(db.EmbeddedDocument):
    meta = {
        'strict': False,
    }
    time = db.DateTimeField(required=True, default=datetime.utcnow)
    value = db.FloatField(required=True)
    currency = db.StringField(max_length=3, required=True)
    variants = db.DictField(required=False, null=True)

    def __unicode__(self):
        try:
            currency = Currency.objects.get(code=self.currency)

            return i18n.stringify(i18n.xformat(
                currency.format_value(self.value, show_cents=False),
                whole=lambda w: w,
                unit=lambda j: j
            ))
        except Exception:
            return u'%s %s' % (self.value, self.currency)

    def __str__(self):
        return self.__unicode__().encode('utf-8')

    @classmethod
    def from_stored(cls, stored_min_price):
        """
        :type stored_min_price: avia.lib.min_price_storage.StoredMinPrice
        :rtype: MinPrice
        """
        return cls(
            value=stored_min_price.value, currency=stored_min_price.currency,
            variants=stored_min_price.stored_variants,
        )


class FilterMinPriceBundle(db.EmbeddedDocument):
    meta = {
        'strict': False,
    }
    filter = db.EmbeddedDocumentField(Filter, required=False, null=False)
    min_prices = db.EmbeddedDocumentListField(MinPrice, required=False, null=False)
    sent_min_price = db.EmbeddedDocumentField(MinPrice)

    def __str__(self):
        return self.to_json()


class Subscription(db.Document):
    meta = {
        'strict': False,
    }
    PRICE_COUNT = 48
    qkey = db.StringField(required=True, primary_key=True)

    point_from_key = db.StringField(required=True)
    point_to_key = db.StringField(required=True)

    date_forward = db.DateTimeField(required=True)
    date_backward = db.DateTimeField(required=False)

    direct_only = db.BooleanField(default=False)

    passengers = db.EmbeddedDocumentField(Passengers)
    national_version = db.StringField(required=True)
    klass = db.StringField(required=True)
    lang = db.StringField(default='ru')

    min_prices = db.EmbeddedDocumentListField(MinPrice)
    sent_min_price = db.EmbeddedDocumentField(MinPrice)
    filtered_minprices = db.EmbeddedDocumentListField(FilterMinPriceBundle, required=False, null=False)

    def __repr__(self):
        return '[Subscription {}] qkey={}, direct_only={}'.format(
            self.id, self.qkey, self.direct_only
        )

    @classmethod
    def from_qkey(cls, qkey):
        """

        :param qkey: c213_c10330_2017-08-19_None_economy_1_0_0_ru
        :return:
        """
        point_from, point_to, date_forward, date_backward, klass, adults, children, infants, nv = qkey.split('_')
        passengers = Passengers(adults=adults, children=children, infants=infants)
        date_forward = datetime.strptime(date_forward, FRONTEND_DATE_FMT)
        date_backward = datetime.strptime(date_backward, FRONTEND_DATE_FMT) if date_backward != 'None' else None

        return cls(
            qkey=qkey,
            point_from_key=point_from,
            point_to_key=point_to,
            date_forward=date_forward,
            date_backward=date_backward,
            passengers=passengers,
            national_version=nv,
            klass=klass,
        )

    @staticmethod
    def get_point_by_key(key):
        """
        :rtype: travel.avia.library.python.common.models_utils.geo.Point
        """
        try:
            return Point.get_any_by_key(key)
        except Exception:
            log.warning('No point with key [%r]', key, exc_info=True)
            return None

    @property
    def point_from(self):
        return self.get_point_by_key(self.point_from_key)

    @property
    def point_to(self):
        return self.get_point_by_key(self.point_to_key)

    def add_filter(self, filter_):
        """
        :param avia.v1.model.filters.Filter filter_:
        :return:
        """
        if filter_:
            used_filters = [bundle.filter for bundle in self.filtered_minprices]
            if filter_ not in used_filters:
                self.filtered_minprices.append(FilterMinPriceBundle(filter=filter_))

    def avia_search_link(self, filter_fragment=None):
        if not filter_fragment:
            filter_fragment = ''

        params = {
            'fromBlock': 'emailAlert',
            'fromId': self.point_from_key,
            'toId': self.point_to_key,
            'when': self.date_forward.strftime(FRONTEND_DATE_FMT),
            'return_date': self.date_backward.strftime(FRONTEND_DATE_FMT) if self.date_backward else '',
            'adult_seats': self.passengers.adults,
            'children_seats': self.passengers.children,
            'infant_seats': self.passengers.infants,
            'klass': self.klass,
            'lang': self.lang,
        }
        host = frontend_host(self.national_version)
        path = 'search'
        if use_travel_portal_links(self.national_version):
            host = travel_frontend_host(self.national_version)
            path = 'avia/search/result/'
        return '{}?{}{}'.format(
            urljoin(host, path),
            urlencode(params),
            filter_fragment
        )

    def order_link(
        self,
        forward_flights=None,
        backward_flights=None,
        filter_fragment=None,
    ):
        """
        Ссылка на страничку с заказом

        :param basestring forward_flights: строка со списком рейсов вида "DP 404.2018-12-27T05:55,DP 406.2018-12-27T22:13"
        :param basestring backward_flights: такая же как и forward_flights строка, но для обратного полёта (если он предусмотрен)
        :rtype: basestring
        """
        if not filter_fragment:
            filter_fragment = ''
        if not forward_flights:
            raise ValueError('No forward_flights string provided for qkey {}'.format(self.qkey))

        if self.date_backward and not backward_flights:
            raise ValueError(
                (
                    'Backward route possible for qkey {}, '
                    'but no backward_flights string provided'
                ).format(self.qkey)
            )

        if not self.date_backward and backward_flights:
            log.warning(
                (
                    'Backward_fligts value is provided ({}), '
                    'but there\'s not backward flight for qkey {}'
                ).format(backward_flights, self.qkey)
            )
        params = {
            'fromBlock': 'emailAlerts',
            'fromId': self.point_from_key,
            'toId': self.point_to_key,
            'adult_seats': self.passengers.adults,
            'children_seats': self.passengers.children,
            'infant_seats': self.passengers.infants,
            'forward': forward_flights,
            'when': self.date_forward.strftime(FRONTEND_DATE_FMT),
            'return_date': self.date_backward.strftime(FRONTEND_DATE_FMT) if self.date_backward else '',
            'backward': backward_flights if backward_flights else '',
        }
        host = frontend_host(self.national_version)
        path = 'order'
        if use_travel_portal_links(self.national_version):
            host = travel_frontend_host(self.national_version)
            path = 'avia/order/'
        return '{}?{}{}'.format(
            urljoin(host, path),
            urlencode(params),
            filter_fragment,
        )

    def is_relevant(self):
        # TODO: localize date_forward and check the relevance more precisely
        return datetime.utcnow().date() <= self.date_forward.date()

    @staticmethod
    def _last_min_prices(subscription):
        by_days = Subscription._min_prices_by_days(subscription.min_prices)
        now = datetime.utcnow().date()
        now_mp = by_days.pop(now, None)
        prev_mp = by_days.get(max(by_days)) if by_days else None
        if subscription.sent_min_price:
            prev_mp = subscription.sent_min_price

        return prev_mp, now_mp

    @staticmethod
    def _min_prices_by_days(min_prices):
        return {
            date: min(group, key=itemgetter('value')) for date, group in
            group_by(min_prices, key=lambda mp: mp.time.date())
        }

    def last_min_prices(self):
        return self._last_min_prices(self)

    def last_filtered_min_price(self, filter_=None):
        """
        :param avia.v1.model.filters.Filter | None filter_:
        :return:
        """
        if not filter_ or filter_ == Filter():
            return self._last_min_prices(self)
        else:
            for bundle in self.filtered_minprices:
                if bundle.filter == filter_:
                    return self._last_min_prices(bundle)
        return None, None

    def update_min_price(self, minprice_provider, popular_provider):
        if not self.point_to:
            log.info('No point_to on subscription(%s)', self)
            return

        if not self.point_from:
            log.info('No point_from on subscription(%s)', self)
            return

        storage = MinPriceYtStorage(
            yt_minprice_variants_provider=minprice_provider,
            yt_popular_variants_provider=popular_provider,
            qkey=self.qkey,
        )

        try:
            price = storage.get_min_price()
        except:
            log.exception('Error getting min price from YT storage')
            price = None

        if not price:
            log.info('No min prices on subscription %r', self)
        else:
            if (
                not self.min_prices or self.min_prices and
                (
                    price.value != self.min_prices[-1].value
                    or price.currency != self.min_prices[-1].currency
                )
            ):
                self.modify(push__min_prices=MinPrice.from_stored(price))
                if len(self.min_prices) > self.PRICE_COUNT:
                    self.modify(pop__min_prices=-1)
                log.info(
                    'Save new min price %d %s on subscription %r',
                    price.value, price.currency, self
                )

        for filter_bundle in list(self.filtered_minprices):
            assert isinstance(filter_bundle, FilterMinPriceBundle)
            price = storage.get_min_price(_filter=filter_bundle.filter)
            log.info('Subscription %r with filter %r', self, filter_bundle.filter)
            if not price:
                log.info('No min prices on subscription %r with filter %r', self, filter_bundle.filter)
                continue
            if (
                not filter_bundle.min_prices or
                filter_bundle.min_prices and
                (
                    price.value != filter_bundle.min_prices[-1].value
                    or price.currency != filter_bundle.min_prices[-1].currency
                )
            ):
                self.modify(
                    query={'filtered_minprices__filter': filter_bundle.filter},
                    push__filtered_minprices__S__min_prices=MinPrice.from_stored(price)
                )
                log.info(
                    'Save new min price %d %s on subscription %r',
                    price.value, price.currency, self
                )
        for filter_bundle in self.filtered_minprices:
            if len(filter_bundle.min_prices) > self.PRICE_COUNT:
                self.modify(
                    query={'filtered_minprices__filter': filter_bundle.filter},
                    pop__filtered_minprices__S__min_prices=-1
                )

        return True


class Relation(db.EmbeddedDocument):
    meta = {
        'strict': False,
    }
    requested_at = db.DateTimeField(null=True)
    """Время запроса привязки"""
    approved_at = db.DateTimeField(null=True)
    """Время подтверждения привязки"""
    approved = db.BooleanField(default=False)
    """Флаг подтвержденности"""

    def make(self):
        self.requested_at = datetime.utcnow()
        log.debug('Constructed Relation: %s', self)
        return self

    def approve(self):
        log.debug('Approving relation')
        self.approved = True
        self.approved_at = datetime.utcnow()

    def __repr__(self):
        return (
            'Relation: '
            '{ requested_at: %s'
            ', approved_at: %s'
            ', approved: %s'
            '}' % (
                self.requested_at,
                self.approved_at,
                self.approved,
            )
        )


class SubscriberSubscription(db.EmbeddedDocument):
    meta = {
        'strict': False,
    }
    """
    Данные о индивидуальном состоянии подписок конкретного подписчика
    """
    created_at = db.DateTimeField(null=False)
    """дата создания подписки"""
    source = db.StringField(null=True)
    """источник подписки (фронт/мобильная версия)"""
    updated_at = db.DateTimeField(null=True)
    """дата обновления подписки"""
    approved = db.BooleanField(default=False)
    """булевый флаг подтверждена ли подписка (либо по клику через 2опт, либо через 1опт)"""
    approved_at = db.DateTimeField(null=True)
    """время подтверждения подписки"""
    pending_passport = db.StringField(null=True)
    """номер паспорта, по которому подписка ожидает подтверждения"""
    pending_session = db.StringField(null=True)
    """номер сессии, по которой подписка ожидает подтверждения"""
    applied_filters = db.EmbeddedDocumentListField(Filter, required=False, null=False)
    """Примененные фильтры"""
    date_range = db.IntField(min_value=1, max_value=120, required=False, default=1)
    """Диапазон дат подписки"""
    last_seen_min_price = db.EmbeddedDocumentField(MinPrice, required=False, null=True)
    """Минимальная цена, которую должен был видеть пользователь в последний раз"""

    def make(self, source, pending_passport_hashed=None, pending_session_hashed=None, date_range=1):
        self.created_at = datetime.utcnow()
        self.date_range = date_range
        self.update_source(source)
        self.update_pending(
            pending_passport_hashed=pending_passport_hashed,
            pending_session_hashed=pending_session_hashed,
        )
        log.debug('Constructed SubscriberSubscription %s', self)
        return self

    def approve(self):
        log.debug('Approving subscription')
        self.approved = True
        self.approved_at = datetime.utcnow()
        self.pending_session = None
        self.pending_passport = None

    def update_pending(self, pending_passport_hashed=None, pending_session_hashed=None):
        log.debug(
            'Updating subscription\'s pending status: pass=%s, sess=%s',
            pending_passport_hashed,
            pending_session_hashed,
        )
        if not self.approved:
            if pending_passport_hashed or pending_session_hashed:
                self.pending_passport = pending_passport_hashed
                self.pending_session = pending_session_hashed
            else:
                self.approve()

    def update_source(self, source, force=False):
        if self.source != source or force:
            self.source = source
            self.updated_at = datetime.utcnow()

    def update_date_range(self, date_range, force=False):
        if self.date_range != date_range or force:
            self.date_range = date_range
            self.updated_at = datetime.utcnow()

    def __repr__(self):
        return (
            'SubscriberSubscription: '
            '{ created_at: %s'
            ', date_range: %s'
            ', source: %s'
            ', updated_at: %s'
            ', approved: %s'
            ', approved_at: %s'
            ', pending_passport: %s'
            ', pending_session: %s'
            '}' % (
                self.created_at,
                self.date_range,
                self.source,
                self.updated_at,
                self.approved,
                self.approved_at,
                self.pending_passport,
                self.pending_session,
            )
        )


class Subscriber(db.Document):
    meta = {
        'indexes': [
            {
                'fields': ['email'],
                'unique': True,
            },
        ],
        'strict': False,
    }
    id = db.UUIDField(primary_key=True, default=uuid.uuid4)
    uid = db.StringField(null=True)
    email = db.EmailField(required=True, null=False)
    subscriptions = db.MapField(db.EmbeddedDocumentField(SubscriberSubscription))
    """
    подписки в виде словаря, где ключ - qkey (идентификатор подписки)
    """
    related_passports = db.MapField(db.EmbeddedDocumentField(Relation))
    """
    словарь с привязанными яндекс-паспортами, которые могут подписываться с 1opt-in
    """
    related_sessions = db.MapField(db.EmbeddedDocumentField(Relation))
    """
    словарь с привязанными сессиями, которые могут подписываться с 1opt-in
    """
    national_version = db.StringField(default='ru')
    language = db.StringField(default='ru')
    currency = db.StringField(default='RUB')
    api_version = db.StringField(null=True)

    def __repr__(self):
        return '[EmailSubscriber {}] subscribed={}, email={}'.format(
            self.id, self.subscribed, self.email
        )

    def __str__(self):
        return '[EmailSubscriber {}] subscribed={}, email={}'.format(
            self.id, self.subscribed, self.email
        )

    @property
    def subscribed(self):
        return len(self.get_approved_subscriptions_list()) > 0

    def add_subscription(
        self, qkey, source,
        filter_=None, pending_passport_plain=None, pending_session_plain=None,
        date_range=1, min_price=None,

    ):
        """
        Добавление подписки к подписчику.

        Подписка не должна отправлятся, пока subscriptions[qkey].approved == False

        :param str qkey: идентификатор подписки
        :param str source: идентификатор источника подписки
        :param avia.v1.model.filters.Filter filter_: фильтр подписки
        :param str pending_passport_plain: необязательное поле. наличие означает,
        что подписка будет ожидать подтверждения, пока пользователь не подтвердит
        связь между Я.паспортом и email.
        :param str pending_session_plain: необязательное поле. наличие означает,
        что подписка будет ожидать подтверждения, пока пользователь не подтвердит
        связь между Я.паспортом и браузерной сессией.
        :param int date_range: диапазон дат подписки
        :param MinPrice min_price: минимальная цена, котрую видел пользователь
        """
        log.info(
            'Subscriber %r. Adding subscription %s %s pending_passport=%s, pending_session=%s, filter=%s',
            self,
            qkey,
            source,
            pending_passport_plain,
            pending_session_plain,
            filter_
        )
        qkey_expanded = self.expand_qkey_daterange(qkey, date_range)
        if qkey in self.subscriptions:
            log.debug(
                'qkey %s in subscriber %s subscriptions list. updating source and pending status',
                qkey,
                self,
            )
            self.subscriptions[qkey].update_source(source)
            self.subscriptions[qkey].update_date_range(date_range)
            self.subscriptions[qkey].update_pending(
                pending_passport_hashed=hashed(pending_passport_plain),
                pending_session_hashed=hashed(pending_session_plain),
            )

        else:
            log.debug(
                'qkey %s not in subscriber %s subscriptions list, creating new one',
                qkey,
                self,
            )
            self.subscriptions[qkey] = SubscriberSubscription().make(
                source,
                pending_passport_hashed=hashed(pending_passport_plain),
                pending_session_hashed=hashed(pending_session_plain),
                date_range=date_range
            )
        if filter_:
            if self.subscriptions[qkey].applied_filters:
                self.subscriptions[qkey].applied_filters[0] = filter_
            else:
                self.subscriptions[qkey].applied_filters.append(filter_)
        else:
            if self.subscriptions[qkey].applied_filters:
                self.subscriptions[qkey].applied_filters = []
        if min_price is not None:
            self.subscriptions[qkey].last_seen_min_price = min_price
        return qkey_expanded

    def expand_approved_subscriptions(self):
        return {
            expanded: qkey
            for qkey in self.get_approved_subscriptions_list()
            for expanded in self.expand_qkey_daterange(qkey, self.subscriptions[qkey].date_range)
        }

    @staticmethod
    def expand_qkey_daterange(qkey, date_range=1, only_relevant=True):
        """
        :param basestring qkey:
        :param int date_range:
        :param bool only_relevant: exapnd only relevant subscriptions, no subscriptions from past
        :rtype: list[basestring]
        """
        qkeys = list()
        assert (1 <= date_range <= 120)
        qkey_struct = structure_from_qkey(qkey)
        for i in xrange(date_range):
            date_forward = qkey_struct.date_forward.date() + timedelta(days=i)
            if only_relevant and date.today() > date_forward:
                continue
            qkeys.append(qkey_from_params(
                point_from_key=qkey_struct.point_from_key,
                point_to_key=qkey_struct.point_to_key,
                date_forward=date_forward,
                date_backward=(
                    qkey_struct.date_backward.date() + timedelta(days=i)
                    if qkey_struct.date_backward else None
                ),
                klass=qkey_struct.klass,
                adults=qkey_struct.adults,
                children=qkey_struct.children,
                infants=qkey_struct.infants,
                nv=qkey_struct.nv,
            ))
        return qkeys

    def actualize_subscriptions(self, subscriptions):
        """
        Удаление всех неактуальных подписок

        :param set[basestring] subscriptions: список актуальных подписок
        :return:
        """
        actual_subscriptions = {}
        for qkey, subscription in self.subscriptions.iteritems():
            for exapnded_qkey in self.expand_qkey_daterange(qkey, subscription.date_range):
                if exapnded_qkey in subscriptions:
                    actual_subscriptions[qkey] = subscription
                    break
        if actual_subscriptions != self.subscriptions:
            self.modify(set__subscriptions=actual_subscriptions)
            log.info('New subscriptions for %r: %s', self, actual_subscriptions)

    def get_approved_subscriptions_list(self):
        """Список подтвержденных подписок"""
        if self.api_version and self.api_version == '1.1':
            log.debug('Api version 1.1. Getting only approved subscriptions.')
            return [
                subscription
                for subscription, content in self.subscriptions.iteritems()
                if content.approved
            ]
        log.debug('Api version OLD. Getting all subscriptions.')
        return list(self.subscriptions.keys())

    def get_pending_subscriptions_list(self, passport_hashed=None, session_hashed=None):
        """
        Список подписок, которые ожидают подтверждения по паспорту или сессии

        :param str passport_hashed: yandex passport
        :param str session_hashed: браузерная сессия
        :return:
        """
        return [
            subscription
            for subscription, content in self.subscriptions.iteritems()
            if (
                passport_hashed and content.pending_passport == passport_hashed
                or session_hashed and content.pending_session == session_hashed
            )
        ]

    def unsubscribe_link(self):
        host = frontend_host(self.national_version)
        path = 'search'
        if use_travel_portal_links(self.national_version):
            host = travel_frontend_host(self.national_version)
            path = 'avia/subscription/unsubscribe'
        url = urljoin(host, path)
        return '{}?{}'.format(url, urlencode({'id': self.id}))

    def unsubscribe_by_direction_link(self, qkey):
        host = frontend_host(self.national_version)
        path = 'subscription/unsubscribe/by_direction'
        if use_travel_portal_links(self.national_version):
            host = travel_frontend_host(self.national_version)
            path = 'avia/subscription/unsubscribe/by_direction'
        url = urljoin(host, path)
        return '{}?{}'.format(url, urlencode({'id': self.id, 'qkey': qkey}))

    def confirm_link(self, passport_plain, session_plain, qkey, filter_fragment=None):
        if not filter_fragment:
            filter_fragment = ''
        url = urljoin(frontend_host(self.national_version),
                      'subscription/confirm')
        args = {
            'id': self.id,
            'utm_source': 'yandex',
            'utm_medium': 'email',
            'utm_campaign': 'price_subscription_confirmation',
            'utm_content': 'redirect',
            'qkey': qkey,
        }
        if passport_plain:
            args['p'] = hashed(passport_plain)
        if session_plain:
            args['s'] = hashed(session_plain)
        return '{}?{}{}'.format(url, urlencode(args), filter_fragment)

    @staticmethod
    def _request_relation(relation_struct, userid_plain):
        """
        Low level request relation operation on relation dict

        :param dict[str, avia.v1.model.subscriber.Relation] relation_struct:
        :param str userid_plain:
        :return:
        """
        if userid_plain:
            uid_hashed = hashed(userid_plain)
            if uid_hashed in relation_struct:
                if not relation_struct[uid_hashed].approved:
                    relation_struct[uid_hashed].requested_at = datetime.utcnow()
            else:
                relation = Relation().make()
                relation.requested_at = datetime.utcnow()
                relation_struct[uid_hashed] = relation

    def request_relation(self, passport_plain=None, session_plain=None):
        """
        Запрос на связку между email и (Я.паспортом или сессией)

        :param str passport_plain: Яндекс паспорт
        :param str session_plain: бразуерная сессия
        :return:
        """
        self._request_relation(self.related_passports, passport_plain)
        self._request_relation(self.related_sessions, session_plain)

    @staticmethod
    def _approve_relation(relation_struct, userid_hashed):
        if userid_hashed and userid_hashed in relation_struct:
            relation_struct[userid_hashed].approve()
            return True
        return False

    def approve_relation(self, passport_hashed=None, session_hashed=None):
        """
        Подтверждение связки между email и (Я.паспортом или сессией)

        :param str passport_hashed: Яндекс паспорт
        :param str session_hashed: браузерная сессия
        :return:
        """
        if passport_hashed:
            self.cleanup_passports()
        if session_hashed:
            self.cleanup_sessions()

        passport_approved = self._approve_relation(self.related_passports, passport_hashed)
        session_approved = self._approve_relation(self.related_sessions, session_hashed)
        return passport_approved or session_approved

    @staticmethod
    def cleanup(relation_struct, must_keep_predicate):
        """
        Очистка связей по предикату.
        Чтобы не занимать память, сначала узнаем нужно ли чистить, потом собираем список связей на удаление,
        затем вычищаем словарь по списку.

        :param dict[str, avia.v1.model.subscriber.Relation] relation_struct: набор паспортов или сессий
        :param function must_keep_predicate: lambda условие, по которому связь должна продолжать жить
        :return: True если очистка привела к изменению размера, иначе False
        """
        for key, relation in relation_struct.items():
            if not must_keep_predicate(relation):
                relation_struct.pop(key)

    def cleanup_passports(self):
        """Чистка списка неподтвержденных паспортов"""
        return self.cleanup(
            self.related_passports,
            lambda relation: (
                relation.approved and datetime.utcnow() - relation.approved_at < APPROVED_PASSPORT_DECAY_TIME
                or datetime.utcnow() - relation.requested_at < UNAPPROVED_SESSION_DECAY_TIME
            )
        )

    def cleanup_sessions(self):
        """Чистка списка неподтвержденных сессий"""
        return self.cleanup(
            self.related_sessions,
            lambda relation: (
                relation.approved and datetime.utcnow() - relation.approved_at < APPROVED_SESSION_DECAY_TIME
                or datetime.utcnow() - relation.requested_at < UNAPPROVED_SESSION_DECAY_TIME
            )
        )

    def get_unapproved_sessions_list(self):
        """
        Cписок неподтвержденных сессий

        :rtype: dict[str, avia.v1.model.subscriber.Relation]
        """
        return [
            session
            for session, relation in self.related_sessions.items()
            if not relation.approved and datetime.utcnow() - relation.requested_at < UNAPPROVED_SESSION_DECAY_TIME
        ]

    def has_too_many_unapproved_sessions(self):
        """
        Флаг слишком много неподтрвежденных сессий

        :rtype: bool
        """
        return len(self.get_unapproved_sessions_list()) >= ALLOWED_UNAPPROVED_SESSIONS

    def session_unban_time(self):
        min_dt = datetime.utcnow()
        for session in self.get_unapproved_sessions_list():
            if self.related_sessions[session].requested_at < min_dt:
                min_dt = self.related_sessions[session].requested_at
        return (min_dt + UNAPPROVED_SESSION_DECAY_TIME).strftime(FRONTEND_DT_FMT)

    def is_2opt_in_progress(self, passport_hashed=None, session_hashed=None):
        """
        Есть ли неподтвержденные сессии для passport или session

        :param passport_hashed: Я.Паспорт
        :param session_hashed: браузерная сессия
        :return:
        """
        if passport_hashed:
            return passport_hashed in self.related_passports and not self.related_passports[passport_hashed].approved
        if session_hashed:
            return session_hashed in self.related_sessions and not self.related_sessions[session_hashed].approved
