# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from datetime import datetime

from travel.avia.contrib.python.mongoengine.mongoengine.queryset.base import PULL

from travel.avia.avia_api.avia.lib.date import datepart
from travel.avia.avia_api.avia.v1.model.db import db
from travel.avia.avia_api.avia.v1.model.favorite import Favorite
from travel.avia.avia_api.avia.v1.model.flight import Flight

log = logging.getLogger(__name__)


class User(db.Document):
    meta = {
        'indexes': [
            {
                'fields': ['yandex_uid', 'uuid'],
                'unique': True,
            }, {
                'fields': ['uuid'],
            }, {
                'fields': ['devices'],
            }, {
                'fields': ['flights'],
            }, {
                'fields': ['aeroex_notify_flights'],
            }
        ],
        'index_background': True,
    }

    devices = db.ListField(
        db.ReferenceField('Device', required=True), default=list
    )

    uuid = db.StringField(required=False, null=True)
    yandex_uid = db.StringField(required=False, null=True)

    flights = db.ListField(
        db.ReferenceField(Flight, reverse_delete_rule=PULL)
    )
    aeroex_notify_flights = db.ListField(
        db.ReferenceField(Flight, reverse_delete_rule=PULL)
    )
    favorites = db.ListField(db.EmbeddedDocumentField(Favorite))
    platforms = db.ListField(db.StringField())

    touched = db.DateTimeField(default=datetime.utcnow)

    def __unicode__(self):
        if self.yandex_uid:
            return 'Y %r %r' % (self.uuid, self.yandex_uid)
        else:
            return 'U %r' % (self.uuid,)

    def __repr__(self):
        return unicode(self)

    @classmethod
    def get_by_appuser(cls, appuser):
        if appuser.yandex_uid:
            cls.objects(
                yandex_uid=appuser.yandex_uid
            ).update(
                unset__uuid=True,
                set__touched=datetime.utcnow(), upsert=True
            )

            user = cls.objects.get(yandex_uid=appuser.yandex_uid)

            user.absorb_users(cls.objects(uuid=appuser.uuid))

            return user

        else:
            cls.objects(uuid=appuser.uuid).update(
                unset__yandex_uid=None,  # По идее должно уже быть None
                set__touched=datetime.utcnow(), upsert=True
            )

            return cls.objects.get(uuid=appuser.uuid)

    def absorb_users(self, users):
        log.info('User %r absorbs users %r', self, users)

        for other in users:
            if other == self:
                continue

            for favorite in other.favorites:
                for existed_favorite in self.iterfind_favorite(
                    favorite.point_from_key, favorite.point_to_key,
                    favorite.date_forward, favorite.date_backward,
                    favorite.direct_only,
                ):
                    log.debug('self.favorites %r', self.favorites)
                    log.debug('pull_favorites %r', existed_favorite)
                    self.update(pull__favorites=existed_favorite)
                    self.reload()
                    log.debug('self.favorites after pull %r', self.favorites)

                self.update(add_to_set__favorites=favorite)

            self.update(add_to_set__flights=other.flights)

            for device in other.devices:
                try:
                    device.assign_to_user_and_save(self)
                except Exception as exc:
                    log.error(
                        "Couldn't absorb device %r from %r to %r: %r",
                        device, other, self, exc
                    )

            self.update(
                add_to_set__aeroex_notify_flights=other.aeroex_notify_flights
            )

            other.delete()
            self.reload()

    def find_favorite(self, point_from_key, point_to_key,
                      date_forward, date_backward, direct_only):
        for favorite in self.iterfind_favorite(
            point_from_key, point_to_key,
            date_forward, date_backward,
            direct_only
        ):
            return favorite

    def iterfind_favorite(self, point_from_key, point_to_key,
                          date_forward, date_backward, direct_only):
        for favorite in self.favorites:
            if all([
                favorite.point_from_key == point_from_key,
                favorite.point_to_key == point_to_key,
                datepart(favorite.date_forward) == datepart(date_forward),
                datepart(favorite.date_backward) == datepart(date_backward),
                favorite.direct_only == direct_only,
            ]):
                yield favorite
