# -*- coding: utf-8 -*-
from django.db import models
from django.utils.translation import ugettext_lazy as _


NATIONAL_VERSIONS_CHOICES = tuple(
    [(v, v) for v in ['ru', 'ua', 'tr']]
)

DAYS_OF_WEEK_CHOICES = tuple(
    [(v, v) for v in range(0, 7)]
)


class MinPrice(models.Model):

    departure_settlement = models.ForeignKey(
        'Settlement',
        verbose_name=_(u'Город отправления'),
        related_name='min_price_departure_settlement'
    )

    arrival_settlement = models.ForeignKey(
        'Settlement',
        verbose_name=_(u'Город прибытия'),
        related_name='min_price_arrival_settlement'
    )

    price = models.IntegerField(
        _(u'Минимальная цена'),
        null=True,
        blank=True
    )

    currency = models.ForeignKey(
        'currency.Currency',
        verbose_name=_(u'Валюта'),
        related_name='min_price_currency',
        null=True
    )

    eventtime = models.DateTimeField(
        verbose_name=_(u'Время события'),
        null=True
    )

    date_forward = models.DateField(
        verbose_name=_(u'Дата вылета'),
        null=True
    )

    date_backward = models.DateField(
        verbose_name=_(u'Дата прилета'),
        null=True
    )

    passengers = models.CharField(
        _(u'Пассажиры'),
        null=True,
        max_length=5
    )

    routes = models.CharField(
        _(u'Номера рейсов'),
        null=True,
        max_length=128
    )

    national_version = models.CharField(
        _(u'Нац. версия'),
        choices=NATIONAL_VERSIONS_CHOICES,
        blank=False,
        default='ru',
        null=False,
        max_length=2
    )

    direct_flight = models.NullBooleanField(
        default=None
    )

    day_of_week = models.PositiveSmallIntegerField(
        null=True,
        choices=DAYS_OF_WEEK_CHOICES,
        default=None
    )

    def __unicode__(self):
        return u'%s - %s (%s%s.)' % (
            self.departure_settlement.title,
            self.arrival_settlement.title,
            self.price, u'р.'
        )

    class Meta:
        verbose_name = _(u'Минимальная цена')
        verbose_name_plural = _(u'Минимальные цены на морде города')
        unique_together = (
            "national_version", "departure_settlement", "arrival_settlement",
            "date_forward", "date_backward", "passengers", "direct_flight"
        )
        app_label = 'www'


class TopFlight(models.Model):
    u"""Топ популярных рейсов"""

    from_point_key = models.CharField(
        blank=False,
        null=False,
        max_length=21,
    )

    to_point_key = models.CharField(
        blank=False,
        null=False,
        max_length=21,
    )

    day_of_week = models.SmallIntegerField(
        null=False,
        blank=False,
        choices=DAYS_OF_WEEK_CHOICES,
    )

    national_version = models.CharField(
        u'Нац. версия',
        choices=NATIONAL_VERSIONS_CHOICES,
        blank=False,
        default='ru',
        null=False,
        max_length=2
    )

    redirects = models.IntegerField(
        null=False,
        blank=False,
    )

    flights = models.CharField(
        u'Рейсы',
        max_length=128,
        blank=False,
        null=False,
    )

    class Meta:
        unique_together = (
            'national_version', 'from_point_key', 'to_point_key',
            'day_of_week', 'redirects', 'flights'
        )

        index_together = [
            ['flights', 'day_of_week', 'redirects'],
        ]

        ordering = (
            'national_version', 'from_point_key', 'to_point_key',
            'day_of_week', 'redirects'
        )
        verbose_name = u'Топ рейс'
        verbose_name_plural = u'Топ популярных рейсов'
        app_label = 'tickets'
