# -*- coding: utf-8 -*-
import logging

from marshmallow import fields

from travel.avia.library.python.common.models.geo import Station

from travel.avia.avia_api.avia.v1.schemas import ApiSchema, CompanySchema

log = logging.getLogger(__name__)


DATE_FORMAT = '%Y-%m-%d'
DT_FORMAT = '%Y-%m-%dT%H:%M%z'


def flight_info_to_key(flight_info):
    return u'{} | {}'.format(
        flight_info.number,
        flight_info.from_station_id,
    )


def segment_to_key(segment):
    return u'{} | {}'.format(
        segment.number,
        segment.departure_station.id
    )


class MinPricesFlightInfoSchema(ApiSchema):
    class Meta:
        ordered = True
        dateformat = DT_FORMAT

    key = fields.Method('get_key', required=True)
    number = fields.String(required=True)
    path_duration = fields.Method('get_path_duration')
    company_id = fields.Integer(default=None)

    departure_key = fields.Method('get_departure_key', required=True)
    arrival_key = fields.Method('get_arrival_key', required=True)

    departure_datetime = fields.DateTime(
        attribute='scheduled_departure_datetime',
    )
    arrival_datetime = fields.DateTime(
        attribute='scheduled_arrival_datetime',
    )

    def get_key(self, data):
        return flight_info_to_key(data)

    def get_departure_key(self, data):
        return 's{}'.format(data.from_station_id)

    def get_arrival_key(self, data):
        return 's{}'.format(data.to_station_id)

    def get_path_duration(self, data):
        departs_at = data.scheduled_departure_datetime
        arrives_at = data.scheduled_arrival_datetime
        return int(
            (arrives_at - departs_at).total_seconds() // 60
        )


class SovetnikMinPricesFlightInfoSchema(MinPricesFlightInfoSchema):
    departure_station = fields.Method('get_departure_station_title', required=True)
    departure_station_iata = fields.Method('get_departure_station_iata', required=True)
    departure_city = fields.Method('get_departure_city', required=True)
    arrival_station = fields.Method('get_arrival_station_title', required=True)
    arrival_station_iata = fields.Method('get_arrival_station_iata', required=True)
    arrival_city = fields.Method('get_arrival_city', required=True)

    def get_departure_station_title(self, data):
        return (
            Station.objects
                .get(pk=data.from_station_id)
                .L_title(lang=self.context['lang'])
        )

    def get_departure_station_iata(self, data):
        return (
            Station.objects
                .get(pk=data.from_station_id)
                .iata
        )

    def get_departure_city(self, data):
        return (
            Station.objects
                .get(pk=data.from_station_id)
                .settlement
                .L_title(lang=self.context['lang'])
        )

    def get_arrival_station_title(self, data):
        return (
            Station.objects
                .get(pk=data.to_station_id)
                .L_title(lang=self.context['lang'])
        )

    def get_arrival_station_iata(self, data):
        return (
            Station.objects
                .get(pk=data.to_station_id)
                .iata
        )

    def get_arrival_city(self, data):
        return (
            Station.objects
                .get(pk=data.to_station_id)
                .settlement
                .L_title(lang=self.context['lang'])
        )


class MinPricesSegmentSchema(ApiSchema):
    class Meta:
        ordered = True
        dateformat = DT_FORMAT

    key = fields.Method('_key', required=True)
    number = fields.String(required=True)
    path_duration = fields.Method('_path_duration')
    company_id = fields.Integer(attribute='company.id', required=False)

    departure_key = fields.String(
        attribute='departure_station.point_key', required=True,
    )
    arrival_key = fields.String(
        attribute='arrival_station.point_key', required=True,
    )

    departure_datetime = fields.DateTime(attribute='departs_at')
    arrival_datetime = fields.DateTime(attribute='arrives_at')

    def _key(self, data):
        return segment_to_key(data)

    def _path_duration(self, segment):
        return (segment.arrives_at - segment.departs_at).total_seconds() // 60


class MinPricesVariantSchema(ApiSchema):
    forward = fields.List(fields.Str(required=True))
    backward = fields.List(fields.Str(required=True))


class MinPricesTariffSchema(ApiSchema):
    price = fields.Decimal(places=2, required=True)
    currency = fields.Str(required=True)


class MinPricesDirectionInfoSchema(ApiSchema):
    variants = fields.List(
        fields.Nested(MinPricesVariantSchema, required=True), required=True
    )
    tariff = fields.Nested(MinPricesTariffSchema, required=True)


class MinPricesEntrySchema(ApiSchema):
    date = fields.Date(required=True, format=DATE_FORMAT)

    direct = fields.Nested(MinPricesDirectionInfoSchema)
    indirect = fields.Nested(MinPricesDirectionInfoSchema)


class MinPricesResultSchema(ApiSchema):
    min_prices = fields.List(
        fields.Nested(MinPricesEntrySchema), required=True
    )

    flights = fields.List(
        fields.Nested(MinPricesFlightInfoSchema), required=True
    )

    companies = fields.List(
        fields.Nested(
            CompanySchema, dump_only=[
                'id', 'title', 'sirena_id', 'iata', 'iata_priority', 'svg_logo'
            ]
        ), required=True
    )

    search_id = fields.Str(required=False)

    class Meta:
        ordered = True


class SovetnikMinPricesResultSchema(MinPricesResultSchema):
    flights = fields.List(
        fields.Nested(SovetnikMinPricesFlightInfoSchema), required=True
    )
    link = fields.String(required=False)


class UpdatedMinPricesSchema(ApiSchema):
    min_prices = fields.List(
        fields.Nested(MinPricesEntrySchema), required=True
    )

    flights = fields.List(
        fields.Nested(MinPricesSegmentSchema), required=True
    )

    companies = fields.List(
        fields.Nested(
            CompanySchema, dump_only=[
                'id', 'title', 'sirena_id', 'iata', 'iata_priority', 'svg_logo'
            ]
        ), required=True
    )

    search_id = fields.Str(required=False)

    class Meta:
        ordered = True
