# -*- coding: utf-8 -*-
import logging
import re

from django.conf import settings
from marshmallow import Schema
from marshmallow.decorators import post_dump

from travel.avia.library.python.common.xgettext.i18n import tgettext

from travel.avia.avia_api.avia.cache.companies import company_cache
from travel.avia.avia_api.ant.custom_types import HttpSearchIdentificator
from travel.avia.avia_api.ant.exceptions import ValidationError
from travel.avia.avia_api.avia.lib.decorators import refuse_None
from travel.avia.avia_api.avia.v1.schemas import fields

log = logging.getLogger(__name__)


class ApiSchema(Schema):
    @post_dump
    def skip_none_values(self, data):
        return type(data)([(k, v) for k, v in data.items() if v is not None])


class FlightStatusSchema(ApiSchema):
    timestamp = fields.DateTime(format='%Y-%m-%d %H:%M:%S UTC')
    code = fields.String()


class FlightSchema(ApiSchema):
    number = fields.String(required=True)
    departure_date = fields.DateTime(format='%Y-%m-%d', required=True)

    canceled = fields.Bool()
    delayed = fields.Bool()
    real_departure = fields.DateTime(format='%Y-%m-%d %H:%M')
    departure_datetime = fields.DateTime(format='%Y-%m-%d %H:%M')


class FlightIdSchema(FlightSchema):
    class Meta:
        fields = ('number', 'departure_date')


class PassengersSchema(ApiSchema):
    adults = fields.Integer()
    children = fields.Integer()
    infants = fields.Integer()

    class Meta:
        ordered = True


class TariffSchema(ApiSchema):
    value = fields.Float(required=True)
    currency = fields.String(required=True)


class CompanySchema(ApiSchema):
    id = fields.Integer(required=True)
    title = fields.Method('get_title')
    iata = fields.String()
    sirena_id = fields.String()
    icao = fields.String()
    icao_ru = fields.String()
    iata_priority = fields.Integer(attribute='priority')

    logo = fields.Method('get_logo')
    svg_logo = fields.Method('get_svg_logo')
    png_logo = fields.Method('get_png_logo')

    def get_title(self, c):
        return c.L_title(lang=self.context.get('lang'))

    def get_logo(self, c):
        fc = company_cache.fallback()

        return (
            media_field_url(c.logo) or
            fc and media_field_url(fc.logo) or None
        )

    def get_svg_logo(self, c):
        fc = company_cache.fallback()

        return (
            media_field_url(c.svg_logo2) or
            fc and media_field_url(fc.svg_logo2)
        )

    def get_png_logo(self, c):
        fc = company_cache.fallback()

        return (
            media_field_url(c.svg2png_logo2) or
            fc and media_field_url(fc.svg2png_logo2)
        )

    class Meta:
        ordered = True


class PartnerSchema(ApiSchema):
    code = fields.String(required=True)
    title = fields.Method('get_title')
    svg_logo = fields.Method('get_svg_logo')
    png_logo = fields.Method('get_png_logo')

    def get_title(self, p):
        return {
            'ru': p.L_national_ru_title,
            'ua': p.L_national_ua_title,
            'kz': p.L_national_kz_title,
            'tr': p.L_national_tr_title,
            'com': (
                p.L_national_com_title
                if hasattr(p, 'L_national_com_title') else
                p.L_national_com
            ),
        }[self.context['national_version']](
            lang=self.context['lang']
        )

    def get_svg_logo(self, p):
        return media_field_url(
            {
                'ru': p.logo2_svg_ru,
                'ua': p.logo2_svg_ua,
                'kz': p.logo2_svg_kz,
                'tr': p.logo2_svg_tr,
                'com': p.logo2_svg_com,
            }[self.context['national_version']]
        )

    def get_png_logo(self, p):
        return media_field_url(
            {
                'ru': p.logo2_svg2png_ru,
                'ua': p.logo2_svg2png_ua,
                'kz': p.logo2_svg2png_kz,
                'tr': p.logo2_svg2png_tr,
                'com': p.logo2_svg2png_com,
            }[self.context['national_version']]
        )


@refuse_None
def media_field_url(field):
    try:
        return field.url

    except Exception:
        pass


class StationSchema(ApiSchema):
    point_key = fields.String()


VALID_FLIGHT_NUMBER_PATTERN = re.compile(
    ur'^(?:[A-ZА-ЯЁ][0-9A-ZА-ЯЁ]|[0-9A-ZА-ЯЁ][A-ZА-ЯЁ])[ -]?[0-9]{1,4}$'
)


class FlightNumberField(fields.String):
    def _deserialize(self, value, attr, data):
        value = super(FlightNumberField, self)._deserialize(value, attr, data)

        return self._validated(value, ValidationError)

    def _validated(self, value, exception_class):
        """ Format value or raise ``exception_class`` if an error occurs. """

        normalized = self._normalize(value)

        try:
            return normalized

        except (TypeError, ValueError) as err:
            raise exception_class(getattr(self, 'error', None) or err)

    def _normalize(self, value):
        value = value.strip().upper()
        valid_match = VALID_FLIGHT_NUMBER_PATTERN.match(value)

        if not valid_match:
            raise ValueError('Bad or empty flight number: %r' % value)

        else:
            val = valid_match.group()

        if val[2] != ' ':
            val = '%s %s' % (val[:2], val[2:])

        return val


class RawFlightSchema(ApiSchema):
    number = FlightNumberField(required=True)
    departure_datetime = fields.DateTime(
        format='%Y-%m-%d %H:%M', required=True
    )
    search_id = fields.Method('encrypt_search_id', 'decrypt_search_id')
    departure = fields.String(required=False)

    def encrypt_search_id(self, obj):
        if 'search_id' in obj:
            return obj['search_id'].encrypt()

    def decrypt_search_id(self, raw):
        try:
            return HttpSearchIdentificator.create_from_crypted(raw)
        except TypeError:
            log.error('Search identificator error: %r' % raw)
            raise ValidationError('Unknown search identificator')


class AeroexpressSchema(ApiSchema):
    station_title = fields.Method('get_station_title')
    title = fields.Method('get_title')
    local_departure_datetime = fields.DateTime(format='%Y-%m-%d %H:%M')
    local_arrival_datetime = fields.DateTime(format='%Y-%m-%d %H:%M')
    price_standart = fields.Float(required=False)
    price_business = fields.Float(required=False)
    price_currency = fields.String(required=False)

    def get_station_title(self, aero):
        return aero.station_point.L_title(lang=self.context.get('lang'))

    def get_title(self, aero):
        lang = self.context.get('lang')
        translation_lang = settings.TRANSLATED_LANGUAGES_MAPPING.get(lang)
        return tgettext(
            u'Аэроэкспресс от <station/>',
            station=aero.station_point.L_popular_title(
                lang=lang, case='genitive'
            ),
            lang=translation_lang,
        )

    class Meta:
        ordered = True


class UserFlightSchema(ApiSchema):
    key = fields.String(attribute='id')

    number = FlightNumberField()
    departure_date = fields.DateTime(format='%Y-%m-%d')
    departure_datetime = fields.DateTime(
        attribute='any_departure_dt', format='%Y-%m-%d %H:%M'
    )
    arrival_datetime = fields.DateTime(
        attribute='arrival_datetime_planed', format='%Y-%m-%d %H:%M'
    )
    path_minutes = fields.Integer(default=None)

    departure = fields.String()
    arrival = fields.String()
    company = fields.Nested(CompanySchema, default=company_cache.fallback)

    aeroexpress_to = fields.Nested(
        AeroexpressSchema, attribute='user_aeroexpress'
    )

    terminal = fields.String()
    gate = fields.String()
    canceled = fields.Bool()
    delayed = fields.Bool()
    checkin_started = fields.Bool()

    class Meta:
        ordered = True


class FlightFromSearchSchema(ApiSchema):
    number = FlightNumberField(required=True)
    departure_datetime = fields.DateTime(
        format='%Y-%m-%d %H:%M', required=True
    )
    departure = fields.String(required=True)


class FlightAddFromSearchResultSchema(ApiSchema):
    flights = fields.Nested(UserFlightSchema, many=True)
    unknown_flights = fields.Nested(FlightFromSearchSchema, many=True)


class FlightListViewSchema(ApiSchema):
    flights = fields.Nested(UserFlightSchema, many=True)


class FlightAddListViewSchema(FlightListViewSchema):
    unknown_flights = fields.Nested(RawFlightSchema, many=True)


class HttpSegmentSchema(ApiSchema):
    number = fields.Str(required=True)
    departure = fields.String(
        attribute='departure_station.point_key', required=True
    )
    arrival = fields.String(
        attribute='arrival_station.point_key', required=True
    )
    departure_datetime = fields.DateTime(
        attribute='departs_at', format='%Y-%m-%d %H:%M'
    )
    arrival_datetime = fields.DateTime(
        attribute='arrives_at', format='%Y-%m-%d %H:%M'
    )
    company_id = fields.Integer(
        attribute='company.id', required=True
    )
    path_minutes = fields.Method('get_path_minutes')

    def get_path_minutes(self, segment):
        return (segment.arrives_at - segment.departs_at).total_seconds() // 60


class HttpTripSchema(ApiSchema):
    key = fields.String(required=True)
    segments = fields.Nested(HttpSegmentSchema, many=True)
    popularity = fields.Constant(0)


class HttpVariantSchema(ApiSchema):
    partner = fields.Function(lambda v: v.partner.code)
    tariff = fields.Nested(TariffSchema)
    original_tariff = fields.Nested(TariffSchema)
    forward = fields.String(attribute='forward.key')
    backward = fields.String(attribute='backward.key')
    vtag = fields.Function(lambda v: v.tag)
    sort_order = fields.Integer()
    ttl = fields.Integer()
    charter = fields.Function(lambda v: v.is_charter or None)
    from_company = fields.Function(lambda v: v.partner.is_aviacompany or None)


class HttpSearchResultsViewSchema(ApiSchema):
    statuses = fields.Raw()
    trips = fields.List(fields.Nested(HttpTripSchema))
    variants = fields.List(fields.Nested(HttpVariantSchema))
    companies = fields.List(fields.Nested(CompanySchema))
    partners = fields.List(fields.Nested(PartnerSchema))
    search_id = fields.String()
