# -*- coding: utf-8 -*-
import logging
from collections import defaultdict
from datetime import timedelta, datetime
from operator import methodcaller
from urllib import urlencode

from django.conf import settings
from django.db.models import F

from travel.avia.library.python.avia_data.models import MinPrice
from travel.avia.library.python.common.models.geo import Settlement
from travel.avia.library.python.common.models_utils.geo import Point

from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.ant.custom_types import (
    Bool, Int, ArgSovetnikDynamicsSettlement, ArgMonth, Str,
    Month,
)
from travel.avia.avia_api.avia.cache.dynamics import dynamics_cache
from travel.avia.avia_api.avia.lib import feature_flag
from travel.avia.avia_api.avia.v1.sovetnik.api import sovetnik_api
from travel.avia.avia_api.avia.v1.sovetnik.helpers import UserInfo

log = logging.getLogger(__name__)


@sovetnik_api.view('/dynamics/')
@sovetnik_api.process_viewparams
def sovetnik_dynamics_handler(
    city_from=ViewParam(name='from', type_=ArgSovetnikDynamicsSettlement()),
    city_to=ViewParam(name='to', type_=ArgSovetnikDynamicsSettlement()),
    month=ViewParam(type_=ArgMonth()),
    one_way=ViewParam(type_=Bool()),
    only_direct=ViewParam(type_=Bool()),
    min_stay_days=ViewParam(type_=Int(), required=False),
    max_stay_days=ViewParam(type_=Int(), required=False),
    user_geoid=ViewParam(type_=Int(), required=False),
    user_ip=ViewParam(type_=Str()),
    clid=ViewParam(type_=Str()),
):
    user_info = UserInfo(user_geoid, user_ip)

    currency = user_info.currency()

    min_prices_qs = MinPrice.objects.filter(
        departure_settlement_id=city_from,
        arrival_settlement_id=city_to,
        national_version=user_info.national_version(),
        currency=currency,
        date_backward__isnull=one_way,
        passengers__endswith='_0_0',  # только взрослые
        date_forward__gte=month.first_day(),
        date_forward__lt=month.plus(1).first_day(),
    ).order_by('id')  # чтобы порядок минцен в дальшейшем всегда был одинаков

    if only_direct:
        min_prices_qs = min_prices_qs.filter(direct_flight=True)

    if not one_way:
        if min_stay_days is not None:
            min_prices_qs = min_prices_qs.filter(
                date_backward__gte=F('date_forward') + timedelta(days=min_stay_days),
            )
        if max_stay_days is not None:
            min_prices_qs = min_prices_qs.filter(
                date_backward__lte=F('date_forward') + timedelta(days=max_stay_days),
            )

    min_prices = tuple(
        PriceWrapper(price_dict) for price_dict in min_prices_qs.values()
    )

    prices_by_date = defaultdict(list)
    for price in min_prices:
        prices_by_date[price.departure_date()].append(price)

    best_prices = {
        date: min(price_group, key=methodcaller('value_per_adult'))
        for date, price_group in prices_by_date.iteritems()
    }.values()  # на каждый день берётся минимальная из цен

    def _serizalized_prices():
        if not best_prices:
            return []
        best_price = min(price.value_per_adult() for price in best_prices)
        return [
            {
                'value': price.value_per_adult(),
                'transfer_count': price.min_transfer_count(),
                'departure_date': price.departure_date(),
                'is_profitable': (price.value_per_adult() / best_price) < 1.2,  # разница не более, чем 20%
                'search_link': price.search_link(clid, only_direct),
            }
            for price in sorted(  # для удобства чтения
                best_prices, key=methodcaller('departure_date')
            )
        ]

    # Месяц отсчёта выбираем текущий, так как надо вернуть цены на год вперёд,
    # но как если бы поиск происходил прямо сейчас.
    now_month = Month.from_datetime(datetime.utcnow())
    year_dynamics = {
        future_month.first_day().strftime('%Y-%m'):
            dynamics_cache.min_price(
                city_from.point_key,
                city_to.point_key,
                now_year=now_month.year(),
                now_month=now_month.month(),
                year=future_month.year(),
                month=future_month.month(),
                currency_code=currency.code,
            )
        for future_month in (
            month.plus(delta) for delta in range(12)
        )
    }

    return {
        'prices': _serizalized_prices(),
        'currency': currency.code,
        'year_dynamics':
            year_dynamics if all(year_dynamics.values()) else None,
    }


class PriceWrapper(object):
    def __init__(self, min_price):
        self._dct = min_price

    def departure_date(self):
        return self._dct['date_forward']

    def value(self):
        return self._dct['price']

    def value_per_adult(self):
        return self._dct['price'] / float(self._dct['passengers'].split('_')[0])

    def min_transfer_count(self):
        return sum(
            min(subroute.count(',') for subroute in part.split(';'))
            for part in self._dct['routes'].split('/')
        )

    def search_link(self, clid, only_direct):
        adults, children, infants = map(int, self._dct['passengers'].split('_'))
        return_date = self._dct['date_backward']

        params = {
            'fromId': Point.get_point_key(
                Settlement, self._dct['departure_settlement_id']
            ),
            'toId': Point.get_point_key(
                Settlement, self._dct['arrival_settlement_id']
            ),
            'adult_seats': adults,
            'children_seats': children,
            'infant_seats': infants,
            'when': self._dct['date_forward'],
            'clid': clid,
        }
        if return_date:
            params['return_date'] = return_date

        if feature_flag.sovetnik_travel_search_switch():
            front_url = settings.TRAVEL_URLS[self._dct['national_version']] + '/search/result/'
        else:
            front_url = settings.FRONTEND_URLS[self._dct['national_version']] + '/search/'

        link = '{url}?{params}'.format(
            url=front_url,
            params=urlencode(params),
        )

        if only_direct:  # todo: проверить, работает ли для портала
            link = '{url}#tt=1,,,0.00,0.00,'.format(url=link)

        return link
