# -*- coding: utf-8 -*-
import logging

from django.conf import settings

from travel.avia.avia_api.avia.cache.top_flights import popularity_cache
from travel.avia.avia_api.avia.daemon_http_api import TicketDaemon
from travel.avia.avia_api.avia.lib.variants_reference import (
    HttpVariantsReference, inject_fictional_companies
)
from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.ant.custom_types import ArgSovetnikSearchId, Int
from travel.avia.avia_api.avia.v1.sovetnik.api import sovetnik_api
from travel.avia.avia_api.avia.v1.sovetnik.schemas import SovetnikHttpSearchResultsSchema

log = logging.getLogger(__name__)


@sovetnik_api.view('/search/results/')
@sovetnik_api.result_schema(SovetnikHttpSearchResultsSchema)
@sovetnik_api.process_viewparams
def sovetnik_http_search_results_handler(
    search_id=ViewParam(type_=ArgSovetnikSearchId()),
    min_ttl=ViewParam(type_=Int(), default=0),
    schema_context=None,
):
    schema_context.update({
        'national_version': search_id.national_version(),
        'lang': search_id.language(),
        'qid': search_id.qid(),
    })

    results = TicketDaemon(settings.DAEMON_URL).search_results(
        qid=search_id.qid(),
        lang=search_id.language(),
        currency=search_id.currency(),
        skip_partners=search_id.partners_to_skip(),
        clid=search_id.clid(),
    )

    variants = results.variants
    inject_fictional_companies(variants)

    smart_variants = SmartVariants.from_variants([
        variant for variant in variants if variant.ttl > min_ttl
    ])
    cheapest = smart_variants.cheapest()
    sold_by_airlines = smart_variants.sold_by_airlines(except_variants=[cheapest])
    most_popular = smart_variants.most_popular(except_variants=[cheapest, sold_by_airlines])

    reference = HttpVariantsReference(
        filter(None, [cheapest, most_popular, sold_by_airlines])
    )

    statuses = results.statuses

    useless_partners = {
        code for code, status in statuses.iteritems() if status == 'done'
    } - {
        cheapest and cheapest.partner.code,
        most_popular and most_popular.partner.code,
        sold_by_airlines and sold_by_airlines.partner.code,
    }

    return {
        'search_is_finished': all(
            status != 'querying'
            for status in statuses.itervalues()
        ),
        'variants': {
            'cheapest': cheapest,
            'most_popular': most_popular,
            'sold_by_airlines': sold_by_airlines,
        },
        'variant_count': search_id.skipped_variant_count() + len(variants),
        'flights': reference.segments(),
        'companies': reference.companies(),
        'partners': reference.partners(),
        'search_id': search_id.with_also_skipped(
            partner_codes=useless_partners,
            variant_count=len([
                v for v in variants if v.partner.code in useless_partners
            ]),
        ).encrypt(),
        'search_params': search_id.query().search_params(),
    }


class SmartVariant(object):
    def __init__(self, variant):
        self._variant = variant

    def variant(self):
        return self._variant

    def popularity(self):
        return (
            popularity_cache.get(self._variant.forward) + (
            popularity_cache.get(self._variant.backward)
                if self._variant.backward else 0
            )
        )

    def is_sold_by_airlines(self):
        return self._variant.partner.is_aviacompany

    def price_key(self):
        def departure_time_priority(dt):
            if not dt:
                return 9
            if 0 <= dt.hour < 6:
                return 3
            if 6 <= dt.hour < 12:
                return 1
            if 12 <= dt.hour < 18:
                return 0
            if 18 <= dt.hour < 24:
                return 2

        return (
            self._variant.tariff['value'],
            (
                self._variant.forward.duration() +
                self._variant.backward.duration()
            ).total_seconds(),
            departure_time_priority(self._variant.forward.starts_at()),
            departure_time_priority(self._variant.backward.starts_at()),
        )


class SmartVariants(object):
    def __init__(self, smart_variant_list):
        self._smarts = tuple(smart_variant_list)

    @classmethod
    def from_variants(cls, variants):
        """

        :type variants: typing.List[avia.http_api_schemas.Variant]
        :rtype: SmartVariants
        """
        return cls([SmartVariant(variant) for variant in variants])

    def cheapest(self):
        if not self._smarts:
            return None

        return min(
            self._smarts, key=lambda smart: smart.price_key()
        ).variant()

    def most_popular(self, except_variants=()):
        if not self._smarts:
            return None

        return SmartVariants([
            smart
            for smart in self._smarts
            if smart.variant() not in except_variants and smart.popularity() > 0
        ]).cheapest()

    def sold_by_airlines(self, except_variants=()):
        return SmartVariants([
            smart
            for smart in self._smarts
            if smart.is_sold_by_airlines() and smart.variant() not in except_variants
        ]).cheapest()
