# coding: utf-8
from datetime import datetime, timedelta
from urllib import urlencode

import pytz
from django.conf import settings
from logging import getLogger
from marshmallow import post_dump

from travel.avia.library.python.common.utils.iterrecipes import pairwise
from travel.avia.library.python.common.xgettext.i18n import tgettext, tngettext

from travel.avia.avia_api.avia.lib import feature_flag
from travel.avia.avia_api.avia.v1.schemas import (
    fields, ApiSchema, TariffSchema, CompanySchema, PartnerSchema
)

DT_FORMAT = '%Y-%m-%dT%H:%M%z'

log = getLogger(__name__)


class SovetnikStationInfoSchema(ApiSchema):
    iata = fields.String(required=False)
    title = fields.Function(
        lambda station, context: station.L_title(lang=context['lang']),
        required=True
    )
    city = fields.Function(
        lambda station, context:
            station.settlement.L_title(lang=context['lang']),
        required=True
    )


def segment_key(segment):
    try:
        return '|'.join([
            segment.number,
            segment.station_from_iata,
            segment.station_to_iata,
            segment.local_departure.strftime('%Y-%m-%dT%H:%M'),
            segment.local_arrival.strftime('%Y-%m-%dT%H:%M'),
        ])
    except AttributeError:
        return segment.key()


class SovetnikSegmentKeySchema(ApiSchema):
    key = fields.Function(segment_key, required=True)

    @post_dump(pass_many=True)
    def convert_to_string(self, data, many):
        try:
            return [item['key'] for item in data]
        except KeyError:
            return [item.key() for item in data]


class SovetnikSegmentSchema(ApiSchema):
    class Meta(object):
        ordered = True
        dateformat = DT_FORMAT

    number = fields.String(required=True)
    key = fields.Function(segment_key, required=True)
    company_id = fields.Int(
        attribute='company.id', required=False  # sometimes a company is unknown
    )
    departure_station = fields.Nested(
        SovetnikStationInfoSchema, attribute='station_from', required=True
    )
    arrival_station = fields.Nested(
        SovetnikStationInfoSchema, attribute='station_to', required=True
    )
    departure_datetime = fields.DateTime(attribute='departure', required=True)
    arrival_datetime = fields.DateTime(attribute='arrival', required=True)


class SovetnikHttpSegmentSchema(SovetnikSegmentSchema):
    key = fields.Function(segment_key, required=True)

    number = fields.Str(required=True)
    departure_station = fields.Nested(SovetnikStationInfoSchema, required=True)
    arrival_station = fields.Nested(SovetnikStationInfoSchema, required=True)
    departure_datetime = fields.DateTime(attribute='departs_at', required=True)
    arrival_datetime = fields.DateTime(attribute='arrives_at', required=True)
    company_id = fields.Int(attribute='company.id', required=True)


class SegmentAdapter(object):
    def __init__(self, http_segment):
        self._s = http_segment

    @property
    def arrival(self):
        return self._s.arrives_at

    @property
    def departure(self):
        return self._s.departs_at

    @property
    def station_from(self):
        return self._s.departure_station

    @property
    def station_to(self):
        return self._s.arrival_station

    @property
    def t_type(self):
        return None  # TODO: В http-вариантах не прописан  t_type


class Transfer(object):
    def __init__(self, arrival_segment, departure_segment):
        self._s1 = (
            SegmentAdapter(arrival_segment)
            if not hasattr(arrival_segment, 't_type')
            else arrival_segment
        )
        self._s2 = (
            SegmentAdapter(departure_segment)
            if not hasattr(departure_segment, 't_type')
            else departure_segment
        )

    def duration(self, lang):
        duration = self._s2.departure - self._s1.arrival

        days = duration.days
        hours, seconds = divmod(duration.seconds, 60 * 60)
        minutes, seconds = divmod(seconds, 60)
        minutes = 5 * (minutes // 5)

        return u' '.join(filter(None, [
            days and tngettext(
                days,
                # В русском все формы одинаковы, разница возникает в белорусском
                u'<days/> д',
                u'<days/> д',
                u'<days/> д',
                days=days, lang=lang,
            ),
            hours and tgettext(
                u'<hours/> ч', hours=hours, lang=lang,
            ),
            minutes and tgettext(
                u'<minutes/> мин', minutes=minutes, lang=lang,
            ),
        ]))

    def type(self):
        if self._s1.t_type != self._s2.t_type:
            return 'transport_change'

        if self._s1.station_to != self._s2.station_from:
            return 'airport_change'

        if self._s1.arrival.date() != self._s2.departure.date():
            return 'night_transfer'

        return 'transfer'

    def text(self, lang):
        stopover = self._s1.station_to
        if stopover.settlement_id:
            stopover = stopover.settlement

        at_place = {  # После обновления коммона phrase_in не работает
            'ru': stopover.title_ru_phrase_in,
            'uk': stopover.title_uk_phrase_in,
        }.get(lang) or stopover.L_title(case='phrase_in', lang=lang)

        return {
            'transport_change': tgettext(
                u'Смена типа транспорта <at_place/>',
                at_place=at_place, lang=lang,
            ),
            'airport_change': tgettext(
                u'Смена аэропорта <at_place/>: <old_airport/> - <new_airport/>',
                at_place=at_place,
                old_airport=self._s1.station_to.L_title(lang=lang),
                new_airport=self._s2.station_from.L_title(lang=lang),
                lang=lang,
            ),
            'night_transfer': tgettext(
                u'Ночная пересадка <at_place/>', at_place=at_place, lang=lang,
            ),
            'transfer': tgettext(
                u'Пересадка <at_place/>', at_place=at_place, lang=lang,
            ),
        }[self.type()]

    def short_text(self, lang):
        return {
            'transport_change': tgettext(u'Смена транспорта', lang=lang),
            'airport_change': tgettext(u'Смена аэропорта', lang=lang),
            'night_transfer': tgettext(u'Ночная пересадка', lang=lang),
            'transfer': tgettext(u'Пересадка', lang=lang),
        }[self.type()]


class SovetnikFlightsSchema(ApiSchema):
    segments = fields.Nested(SovetnikSegmentKeySchema, many=True)
    transfers = fields.Method('_transfers', required=True)

    def _transfers(self, trip):
        return [
            {
                'type': transfer.type(),
                'duration': transfer.duration(self.context['lang']),
                'text': transfer.text(self.context['lang']),
                'short_text': transfer.short_text(self.context['lang']),
            }
            for transfer in (
                Transfer(s1, s2) for s1, s2 in pairwise(trip.segments)
            )
        ]


class SovetnikPartnerSchema(PartnerSchema):
    class Meta(object):
        fields = ('code', 'title', 'svg_logo')


class SovetnikVariantSchema(ApiSchema):
    forward = fields.Nested(
        SovetnikSegmentKeySchema, attribute='forward.segments', many=True,
        required=True
    )
    backward = fields.Nested(
        SovetnikSegmentKeySchema, attribute='backward.segments', many=True
    )
    new_forward = fields.Nested(
        SovetnikFlightsSchema, attribute='forward', required=True
    )
    new_backward = fields.Nested(SovetnikFlightsSchema, attribute='backward')
    tariff = fields.Nested(
        TariffSchema, attribute='national_tariff', required=True
    )
    partner_code = fields.String(attribute='partner.code', required=True)
    link = fields.Method('_link', required=True)
    expires_at = fields.Method('_expires_at', required=True)

    def _link(self, variant):
        lang = self.context['lang']
        qid = self.context['qid']

        if feature_flag.sovetnik_travel_redirect_switch():
            front_url = settings.TRAVEL_URLS[self.context['national_version']] + '/redirect/'
        else:
            front_url = settings.FRONTEND_URLS[self.context['national_version']] + '/order/redirect/'

        return (
            front_url + '?' + urlencode({
                'tag': variant.tag,
                'p': variant.partner.code,
                'qid': qid,
                'q': qid,
                'lang': lang,
                'clid': self.context['clid'],
            })
        )

    def _expires_at(self, variant):
        return pytz.UTC.localize(
            datetime.utcnow() + timedelta(seconds=variant.ttl)
        ).strftime('%Y-%m-%dT%H:%M:%S%z')


class SovetnikHttpVariantSchema(SovetnikVariantSchema):
    forward = fields.Nested(
        SovetnikSegmentKeySchema, attribute='forward.segments', many=True,
        required=True
    )
    backward = fields.Nested(
        SovetnikSegmentKeySchema, attribute='backward.segments', many=True
    )
    new_forward = fields.Nested(SovetnikFlightsSchema, attribute='forward', required=True)
    new_backward = fields.Nested(SovetnikFlightsSchema, attribute='backward')
    tariff = fields.Nested(TariffSchema, required=True)
    expires_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S%z', required=True)
    link = fields.Str(attribute='deep_link', required=True)
    order_link = fields.Str(required=True)


class SovetnikCheapestVariantSchema(SovetnikVariantSchema):
    text = fields.Function(lambda obj, context: tgettext(
        u'Лучшая цена', lang=context['lang']
    ), required=True)


class SovetnikCheapestHttpVariantSchema(SovetnikHttpVariantSchema):
    text = fields.Function(lambda obj, context: tgettext(
        u'Лучшая цена', lang=context['lang']
    ), required=True)


class SovetnikMostPopularVariantSchema(SovetnikVariantSchema):
    text = fields.Function(lambda obj, context: tgettext(
        u'Самый популярный', lang=context['lang']
    ), required=True)


class SovetnikMostPopularHttpVariantSchema(SovetnikHttpVariantSchema):
    text = fields.Function(lambda obj, context: tgettext(
        u'Самый популярный', lang=context['lang']
    ), required=True)


class SovetnikAirlinesVariantSchema(SovetnikVariantSchema):
    text = fields.Function(lambda obj, context: tgettext(
        u'Продаёт авиакомпания', lang=context['lang']
    ), required=True)


class SovetnikAirlinesHttpVariantSchema(SovetnikHttpVariantSchema):
    text = fields.Function(lambda obj, context: tgettext(
        u'Продаёт авиакомпания', lang=context['lang']
    ), required=True)


class SovetnikVariantsSchema(ApiSchema):
    cheapest = fields.Nested(
        SovetnikCheapestVariantSchema, required=True, default=None
    )
    most_popular = fields.Nested(
        SovetnikMostPopularVariantSchema, required=True, default=None
    )
    sold_by_airlines = fields.Nested(
        SovetnikAirlinesVariantSchema, required=True, default=None
    )


class SovetnikHttpVariantsSchema(ApiSchema):
    cheapest = fields.Nested(
        SovetnikCheapestHttpVariantSchema, required=True, default=None
    )
    most_popular = fields.Nested(
        SovetnikMostPopularHttpVariantSchema, required=True, default=None
    )
    sold_by_airlines = fields.Nested(
        SovetnikAirlinesHttpVariantSchema, required=True, default=None
    )


class SovetnikCompanySchema(CompanySchema):
    class Meta(object):
        fields = ('id', 'title', 'svg_logo')


class SovetnikSearchResultsSchema(ApiSchema):
    class Meta(object):
        ordered = True

    search_is_finished = fields.Boolean(required=True)
    variants = fields.Nested(SovetnikVariantsSchema, required=True)
    variant_count = fields.Integer(required=True)
    flights = fields.Nested(SovetnikSegmentSchema, required=True, many=True)
    companies = fields.Nested(SovetnikCompanySchema, required=True, many=True)
    partners = fields.Nested(SovetnikPartnerSchema, required=True, many=True)
    search_id = fields.String(required=True)


class SovetnikStationSchema(ApiSchema):
    iata = fields.String(required=False)
    title = fields.Function(
        lambda station, context: station.L_title(lang=context['lang']),
        required=True
    )


class SovetnikSettlementSchema(ApiSchema):
    iata = fields.String(required=False)
    title = fields.Function(
        lambda station, context: station.L_title(lang=context['lang']),
        required=True
    )


class SovetnikPointSchema(ApiSchema):
    settlement = fields.Nested(SovetnikSettlementSchema, required=True)
    station = fields.Nested(SovetnikStationSchema, required=False)
    substitution = fields.Boolean(required=False)


class SovetnikDirectionSchema(ApiSchema):
    departure = fields.Nested(SovetnikPointSchema, required=True)
    arrival = fields.Nested(SovetnikPointSchema, required=True)


class SovetnikSearchResponseSchema(ApiSchema):
    search_id = fields.String(required=True)
    link = fields.String(required=True)
    direction = fields.Nested(SovetnikDirectionSchema, required=True)
    user_country = fields.String(required=True)
    currency = fields.String(required=True)


class SovetnikHttpSearchParamsSchema(ApiSchema):
    direction = fields.Nested(SovetnikDirectionSchema, required=True)
    national_version = fields.String(required=True)
    passengers = fields.Dict(required=True)
    date_forward = fields.String(required=True)
    date_backward = fields.String(required=False)


class SovetnikHttpSearchResultsSchema(SovetnikSearchResultsSchema):
    variants = fields.Nested(SovetnikHttpVariantsSchema, required=True)
    flights = fields.Nested(SovetnikHttpSegmentSchema, required=True, many=True)
    search_params = fields.Nested(SovetnikHttpSearchParamsSchema, required=True)
