# -*- coding: utf-8 -*-
import logging
from datetime import timedelta
from urllib import urlencode

from django.conf import settings

from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.ant.custom_types import (
    SovetnikSearchIdentificator, Str, ArgLang, Int, Date, ArgSovetnikPoint,
    ArgsList, ArgCurrency,
)
from travel.avia.avia_api.ant.exceptions import ValidationError
from travel.avia.avia_api.avia.api.daemon_api import DaemonQuery
from travel.avia.avia_api.avia.daemon_http_api import TicketDaemon, DaemonBadRequest, LoggingTD
from travel.avia.avia_api.avia.lib.flight_classes import FlightClass
from travel.avia.avia_api.avia.lib import feature_flag
from travel.avia.avia_api.avia.v1.sovetnik.helpers import UserInfo, substitution_reference
from travel.avia.avia_api.avia.v1.sovetnik.schemas import SovetnikSearchResponseSchema
from travel.avia.avia_api.avia.v1.sovetnik.api import sovetnik_api

log = logging.getLogger(__name__)


@sovetnik_api.view('/search/')
@sovetnik_api.result_schema(SovetnikSearchResponseSchema)
@sovetnik_api.process_viewparams
def sovetnik_search_handler(
    point_from=ViewParam(name='from', type_=ArgSovetnikPoint(), required=False),
    point_to=ViewParam(name='to', type_=ArgSovetnikPoint()),
    departure_date=ViewParam(type_=Date()),
    return_date=ViewParam(type_=Date(), required=False),
    adults=ViewParam(type_=Int([1, 9]), default=1),
    children=ViewParam(type_=Int([0, 9]), default=0),
    infants=ViewParam(type_=Int([0, 9]), default=0),
    without=ViewParam(type_=ArgsList(Str()), default=tuple()),
    currency=ViewParam(type_=ArgCurrency(), required=False),
    lang=ViewParam(type_=ArgLang()),
    user_geoid=ViewParam(type_=Int(), required=False),
    user_ip=ViewParam(type_=Str(), required=False),
    clid=ViewParam(type_=Str()),
    yaclid=ViewParam(type_=Str(), required=False),
    schema_context=None,
):
    schema_context['lang'] = lang

    daemon = TicketDaemon(settings.DAEMON_URL)

    if yaclid:
        try:
            age = daemon.yaclid_age(yaclid)
        except DaemonBadRequest as exc:
            # При некорректных яклидах ничего не делаем - защита от спуфинга
            log.warning('Malformed yaclid: %r (%r)', yaclid, exc)
        else:
            if age < timedelta(minutes=30):
                raise ValidationError('Yaclid is too young: {}'.format(age))

    user_info = UserInfo(user_geoid, user_ip)
    if not currency:
        currency = user_info.currency()

    point_from = point_from or user_info.settlement
    if not point_from:
        raise ValidationError('Unable to determine departure point')

    point_from_reference = substitution_reference(point_from)
    point_to_reference = substitution_reference(point_to)

    point_from = point_from_reference.point()
    point_to = point_to_reference.point()

    try:
        qid = LoggingTD(daemon).init_search(
            point_from=point_from, point_to=point_to,
            date_forward=departure_date,
            date_backward=return_date,
            adults=adults, children=children, infants=infants,
            flight_class=FlightClass.Economy,
            national_version=user_info.national_version(),
            lang=lang,
            country_geo_id=None,
            service='sovetnik',
            user_ip=user_ip,
            user_geoid=user_geoid,
        )
    except DaemonBadRequest as exc:
        raise ValidationError('Daemon bad request: {}'.format(exc))

    if feature_flag.sovetnik_travel_search_switch():
        front_url = settings.TRAVEL_URLS[user_info.national_version()] + '/search/result/'
    else:
        front_url = settings.FRONTEND_URLS[user_info.national_version()] + '/search/'

    return {
        'search_id': SovetnikSearchIdentificator(
            daemon_query=DaemonQuery.from_search_args(
                point_from=point_from,
                point_to=point_to,
                passengers={
                    'adults': adults,
                    'children': children,
                    'infants': infants
                },
                klass=FlightClass.Economy,
                date_forward=departure_date,
                date_backward=return_date,
                national_version=user_info.national_version(),
            ),
            qid=qid,
            partners_to_skip=without,
            currency=currency.code,
            lang=lang,
            clid=clid,
            national_version=user_info.national_version(),
        ).encrypt(),
        'link': (
            front_url +
            '?' +
            urlencode(dict(
                fromId=point_from.point_key,
                toId=point_to.point_key,
                adult_seats=adults,
                children_seats=children,
                infant_seats=infants,
                when=departure_date,
                clid=clid,
                **(dict(
                    return_date=return_date
                ) if return_date else {})
            ))
        ),
        'direction': {
            'departure': point_from_reference,
            'arrival': point_to_reference,
        },
        'user_country': user_info.country().L_title(lang=lang),
        'currency': currency.code,
    }
