# -*- coding: utf-8 -*-
import logging
import operator
from datetime import timedelta, datetime

import pytz
from django.db.models import Q

from travel.avia.library.python.avia_data.models import TopFlight

from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.ant.custom_types import ArgSovetnikSearchId, Int
from travel.avia.avia_api.avia.lib.variants_reference import VariantsReference
from travel.avia.avia_api.avia.v1.sovetnik.api import sovetnik_api
from travel.avia.avia_api.avia.v1.sovetnik.schemas import SovetnikSearchResultsSchema

log = logging.getLogger(__name__)


@sovetnik_api.view('/search/old_results/')
@sovetnik_api.result_schema(SovetnikSearchResultsSchema)
@sovetnik_api.process_viewparams
def sovetnik_search_results_handler(
    search_id=ViewParam(type_=ArgSovetnikSearchId()),
    min_ttl=ViewParam(type_=Int(), default=0),
    schema_context=None,
):
    query = search_id.query()

    schema_context.update({
        'national_version': query.national_version,
        'lang': search_id.language(),
        'qid': search_id.qid(),
        'clid': search_id.clid(),
    })

    daemon_result = query.result(
        skip_partners=search_id.partners_to_skip(),
        currency=search_id.currency(),
    )

    variants = daemon_result.available_variants(
        aware_now=pytz.UTC.localize(datetime.utcnow())
    )

    smart_variants = SmartVariants.from_variants([
        variant for variant in variants if variant.ttl > min_ttl
    ])
    cheapest = smart_variants.cheapest()
    most_popular = smart_variants.most_popular()
    sold_by_airlines = smart_variants.sold_by_airlines()

    reference = VariantsReference(
        filter(None, [cheapest, most_popular, sold_by_airlines])
    )

    statuses = daemon_result.statuses()

    useless_partners = {
        code for code, status in statuses.iteritems() if status == 'done'
    } - {
        cheapest and cheapest.partner.code,
        most_popular and most_popular.partner.code,
        sold_by_airlines and sold_by_airlines.partner.code,
    }

    return {
        'search_is_finished': all(
            status != 'querying'
            for status in statuses.itervalues()
        ),
        'variants': {
            'cheapest': cheapest,
            'most_popular': most_popular,
            'sold_by_airlines': sold_by_airlines,
        },
        'variant_count': search_id.skipped_variant_count() + len(variants),
        'flights': reference.segments(),
        'companies': reference.companies(),
        'partners': reference.partners(),
        'search_id': search_id.with_also_skipped(
            partner_codes=useless_partners,
            variant_count=len([
                v for v in variants if v.partner.code in useless_partners
            ]),
        ).encrypt(),
    }


class Popularities(object):
    def __init__(self, trips):
        queries = [
            Q(
                from_point_key=trip.station_from.settlement.point_key,
                to_point_key=trip.station_to.settlement.point_key,
                day_of_week=trip.local_departure.weekday(),
                flights=trip.key().replace('-', ';')
            )
            for trip in trips
            if trip.station_from.settlement and trip.station_to.settlement
        ]
        if queries:
            self._redirects = {
                (
                    top.from_point_key,
                    top.to_point_key,
                    top.day_of_week,
                    top.flights
                ): top.redirects
                for top in TopFlight.objects.filter(
                    reduce(operator.or_, queries)
                ).order_by('redirects')
            }
        else:
            self._redirects = {}

    def get(self, trip):
        if trip.station_from.settlement and trip.station_to.settlement:
            return self._redirects.get(
                (
                    trip.station_from.settlement.point_key,
                    trip.station_to.settlement.point_key,
                    trip.local_departure.weekday(),
                    trip.key().replace('-', ';')
                ), 0
            )
        return 0


class SmartVariant(object):
    def __init__(self, variant, popularities):
        self._variant = variant
        self._popularities = popularities

    def variant(self):
        return self._variant

    def popularity(self):
        return (
            self._popularities.get(self._variant.forward) + (
                self._popularities.get(self._variant.backward)
                if self._variant.backward else 0
            )
        )

    def is_sold_by_airlines(self):
        return self._variant.partner.is_aviacompany

    def price_key(self):
        def departure_time_priority(dt):
            if not dt:
                return 9
            if 0 <= dt.hour < 6:
                return 3
            if 6 <= dt.hour < 12:
                return 1
            if 12 <= dt.hour < 18:
                return 0
            if 18 <= dt.hour < 24:
                return 2

        return (
            int(self._variant.national_tariff.base_value or 0),
            int(self._variant.national_tariff.value or 0),
            (
                self._variant.forward.duration +
                (self._variant.backward.duration or timedelta(0))
            ).total_seconds(),
            departure_time_priority(self._variant.forward.local_departure),
            departure_time_priority(self._variant.backward.local_departure),
        )


class SmartVariants(object):
    def __init__(self, smart_variant_list):
        self._smarts = tuple(smart_variant_list)

    @classmethod
    def from_variants(cls, variants):
        popularities = Popularities(VariantsReference(variants).trips())
        return cls([
            SmartVariant(variant, popularities)
            for variant in variants
        ])

    def cheapest(self):
        if not self._smarts:
            return None
        return min(
            self._smarts, key=lambda smart: smart.price_key()
        ).variant()

    def most_popular(self):
        cheapest = self.cheapest()
        if not cheapest:
            return None
        return SmartVariants([
            smart
            for smart in self._smarts
            if smart.variant().tag != cheapest.tag
            and smart.popularity() > 0
        ]).cheapest()

    def sold_by_airlines(self):
        return SmartVariants([
            smart
            for smart in self._smarts
            if smart.is_sold_by_airlines()
        ]).cheapest()
