# -*- coding: utf-8 -*-
import logging
from functools import wraps
from itertools import ifilter, imap
from operator import attrgetter

import ujson
from django.conf import settings
from flask import Blueprint, request
from travel.avia.contrib.python.mongoengine.mongoengine import DoesNotExist
from werkzeug.exceptions import Forbidden, Unauthorized, BadRequest

from ticket_parser2.api.v1.exceptions import TicketParsingException
from travel.library.python.tvm_ticket_provider import provider_fabric

import travel.avia.avia_api.ant.api_interface
from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.ant.custom_types import Int
from travel.avia.avia_api.avia.cache.settlements import settlement_cache
from travel.avia.avia_api.avia.lib.jsend import BaseJsend
from travel.avia.avia_api.avia.lib.passport_utils import get_emails_by_uid
from travel.avia.avia_api.avia.v1.model.subscriber import Subscriber, Subscription
from travel.avia.avia_api.avia.v1.model.user import User

takeout_blueprint = Blueprint('takeout_v1', __name__)
logger = logging.getLogger(__name__)

HUMAN_READABLE_DATETIME = '%Y-%m-%d %H:%M:%S'


def tvm_auth(allowed_service_ids):
    tvm_provider = provider_fabric.create(
        fake=not settings.ENABLE_TVM,
        renew=True,
    )

    def wrapped_tvm_auth(fn):
        @wraps(fn)
        def decorated(*args, **kwargs):
            if not settings.ENABLE_TVM:
                return fn(*args, **kwargs)

            if 'X-Ya-Service-Ticket' not in request.headers:
                raise BadRequest('Empty tvm service ticket')

            try:
                ticket = tvm_provider.check_service_ticket(request.headers['X-Ya-Service-Ticket'])
            except TicketParsingException as e:
                message = 'TvmError: %s', e.message
                logger.error(message)
                raise Unauthorized(message)

            request_service_id = ticket.src
            if request_service_id not in allowed_service_ids:
                raise Forbidden('Access for service=%r is not allowed' % request_service_id)

            return fn(*args, **kwargs)

        return decorated

    return wrapped_tvm_auth


takeout_api = travel.avia.avia_api.ant.api_interface.Ant(
    takeout_blueprint,
    auth_decorator=tvm_auth(allowed_service_ids=[settings.TVM_TAKEOUT_ID]),
)


def takeout_flight(flight):
    """

    :type flight: avia_api.avia.v1.model.flight.Flight
    :rtype: dict
    """
    return {
        'number': flight.number,
        'departure_date': str(flight.departure_date.date()),
    }


def get_title(point_key):
    point_id = int(point_key[1:])
    settlement = settlement_cache.by_id(point_id)
    if settlement:
        return settlement.title
    station_settlements = settlement_cache.by_station_id(point_id)
    if station_settlements:
        return next(ifilter(None, imap(attrgetter('title'), station_settlements)), None)


def takeout_favorite(favorite):
    """

    :type favorite: avia_api.avia.v1.model.user.Favorite
    :rtype: dict
    """

    return {
        'point_from_key': favorite.point_from_key,
        'point_from_title': get_title(favorite.point_from_key),
        'point_to_key': favorite.point_to_key,
        'point_to_title': get_title(favorite.point_to_key),
        'forward_date': str(favorite.date_forward.date()),
        'backward_date': str(favorite.date_backward.date()) if favorite.date_backward else None,
        'passengers': {
            'adults': favorite.passengers.adults,
            'children': favorite.passengers.children,
            'infants': favorite.passengers.infants,
        }
    }


@takeout_api.view('/mobile_api_user', methods=['POST'], disable_auth_in_environments=['development'])
def takeout_mobile_api_user_handler(
    uid=ViewParam(name='uid', type_=Int(), required=True),
    unixtime=ViewParam(name='unixtime', type_=Int(), required=True),
):
    logger.info('/mobile_api_user takeout request uid=%d', uid)
    try:
        user = User.objects.get(yandex_uid=str(uid))
    except DoesNotExist:
        return BaseJsend(
            status_code=200,
            status='no_data'
        )

    flights = map(takeout_flight, user.flights)
    favorites = map(takeout_favorite, user.favorites)
    return BaseJsend(
        status_code=200,
        status='ok',
        data={
            'flights.json': ujson.dumps(flights, ensure_ascii=False),
            'favorites.json': ujson.dumps(favorites, ensure_ascii=False),
        },
    )


# Subscriptions takeout


def takeout_subscription(email, subscription, subscriber_subscription):
    """
    :param avia.v1.model.subscriber.Subscription subscription:
    :param avia.v1.model.subscriber.SubscriberSubscription subscriber_subscription:
    :rtype: dict
    """
    return {
        'email': email,
        'point_from_key': subscription.point_from_key,
        'point_from_title': get_title(subscription.point_from_key),
        'point_to_key': subscription.point_to_key,
        'point_to_title': get_title(subscription.point_to_key),
        'date_forward': str(subscription.date_forward.date()),
        'date_backward': str(subscription.date_backward.date()) if subscription.date_backward else None,
        'national_vresion': subscription.national_version,
        'klass': subscription.klass,
        'lang': subscription.lang,
        'created_at': subscriber_subscription.created_at.strftime(HUMAN_READABLE_DATETIME),
        'approved': subscriber_subscription.approved,
        'approved_at': (
            subscriber_subscription.approved_at.strftime(
                HUMAN_READABLE_DATETIME
            ) if subscriber_subscription.approved_at else None
        ),
    }


@takeout_api.view('/subscriptions', methods=['POST'], disable_auth_in_environments=['development'])
def takeout_subscriptions_user_handler(
    uid=ViewParam(name='uid', type_=Int(), required=True),
    unixtime=ViewParam(name='unixtime', type_=Int(), required=True),
):
    logger.info('/subscriptions takeout request uid=%d', uid)
    subscribers = []
    for email in get_emails_by_uid(uid):
        try:
            subscribers.append(Subscriber.objects.get(email=email))
        except DoesNotExist:
            continue

    if not subscribers:
        logger.info('User %d does not have any emails subscribed', uid)
        return BaseJsend(
            status_code=200,
            status='no_data'
        )

    subscriptions = []
    for subscriber in subscribers:
        for qkey, subscriber_subscription in subscriber.subscriptions.items():
            try:
                subscription = Subscription.objects.get(qkey=qkey)

            except DoesNotExist:
                continue

            try:
                subscriptions.append(takeout_subscription(
                    email=subscriber.email,
                    subscription=subscription,
                    subscriber_subscription=subscriber_subscription,
                ))
            except AttributeError:
                logger.exception('Invalid subscription %r content', subscription)

    if not subscriptions:
        logger.info('User %d does not have any relevant subscriptions', uid)
        return BaseJsend(
            status_code=200,
            status='no_data'
        )

    logger.info('User %d gets his %d subscriptions', uid, len(subscriptions))
    return BaseJsend(
        status_code=200,
        status='ok',
        data={
            'subscriptions.json': ujson.dumps(subscriptions, ensure_ascii=False),
        },
    )
