# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from collections import OrderedDict

from django.conf import settings
from flask import request, session
from flask_login import (
    login_user, login_required, current_user, logout_user
)

from travel.avia.library.python.common.geotargeting.lookup import get_city_by_ip

import travel.avia.avia_api.ant.custom_types
from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.ant.argparser import LoggingRawArgParser
from travel.avia.avia_api.ant.custom_types import ArgAppUserByAppKey, ArgUUID, DateTimeGmt
from travel.avia.avia_api.ant.iso_types import (
    IsoCountryArg, IsoLocaleArg, IsoLanguageArg, IsoCurrencyArg, IsoLocale,
)
from travel.avia.avia_api.avia.business.hello import hello
from travel.avia.avia_api.avia.lib.decorators import skip_None_values
from travel.avia.avia_api.avia.lib.pusher import Pusher
from travel.avia.avia_api.avia.lib.remote_addr import real_client_ip
from travel.avia.avia_api.avia.v1.views.base import api, api_v2

log = logging.getLogger(__name__)


@api.view('/hello/',
          methods=['GET', 'POST'] if settings.DEBUG else ['POST'],
          sort_order=0.1,
          argparser=LoggingRawArgParser(logging.getLogger('hello')))
@skip_None_values
def hello_v1(
    push_token=ViewParam(required=False),
    uuid=ViewParam(type_=ArgUUID()),
    oauth_token=ViewParam(required=False),
    appuser=ViewParam(name='app_key', type_=ArgAppUserByAppKey()),
    timestamp=ViewParam(type_=DateTimeGmt()),
    national_version=ViewParam(
        choices=[
            version
            for version in settings.NATIONAL_VERSIONS
            if version in ['ru', 'ua']
        ],
        about='Национальная версия'
    ),
):
    """ Hello: Начало сессии """

    # RASPTICKETS-4957 - app_key logging is required
    log.info(
        'Hello_v1 request: uuid=%r, oauth_token=%r, app_key=%s, '
        'push_token=%r, appuser: %r',
        uuid, oauth_token, appuser.app_key, push_token, appuser
    )

    login_user(appuser)

    new_session = hello(
        session=session,
        uuid=uuid,
        oauth_token=oauth_token,
        push_token=push_token,
        platform=appuser.platform,
        device_time=timestamp,
        lang={
            'ru': 'ru',
            'ua': 'uk',
        }[national_version],
        country={
            'ru': 'RU',
            'ua': 'UA',
        }[national_version],
        currency={
            'ru': 'RUR',
            'ua': 'UAH',
        }[national_version],
        national_version=national_version,
        remote_address=real_client_ip(request),
        known_device_id=session.get('device_id')
    )

    ip_city, _ = get_city_by_ip(real_client_ip(request))

    session.clear()
    session.update(new_session)

    return {
        'ttl_seconds': appuser.session_ttl,
        'X-SESSION': session.get_xsession_value(),
        'oauth_token_invalid': (oauth_token and not session['yandex_uid']),
        'client_city': ip_city.point_key if ip_city else None,
    }


@api.view('/bye/', sort_order=0.2)
@login_required
def bye():
    """ Bye: Конец сессии """

    uid = current_user.yandex_uid
    uuid = current_user.uuid

    log.info('Bye request: %(uid)s, %(uuid)s', dict(uid=uid, uuid=uuid))

    Pusher().delete_by_uuid(uuid)

    logout_user()
    session.clear()

    return True


class GuessedSettings(object):
    def __init__(self, country, language, currency,
                 locale, ip_country, settings_country,
                 sim_countries, gps_country,
                 known_languages, known_currencies):

        self._country = country
        self._lang = language
        self._currency = currency

        self._locale = locale

        self._ip_country = ip_country
        self._settings_country = settings_country
        self._sim_countries = sim_countries or []
        self._gps_country = gps_country
        self._known_languages = known_languages
        self._known_currencies = known_currencies

    def possible_currencies(self):
        return tuple(
            currency
            for currency in {
                'ru': ('RUR', 'USD', 'EUR'),
                'ua': ('UAH', 'USD', 'EUR'),
                'kz': ('KZT', 'RUR', 'USD', 'EUR'),
                'tr': ('TRY', 'USD', 'EUR'),
                'com': ('EUR', 'USD', 'GBP'),
            }[self.national_version()]
            if currency in self._known_currencies
        )

    def chosen_currency(self):
        choices = self.possible_currencies()
        return self._currency if self._currency in choices else choices[0]

    def possible_languages(self):
        return tuple(
            lang
            for lang in settings.AVIA_API_LANGUAGES
            if lang in self._known_languages
        )

    def chosen_language(self):
        lang = self._lang or self._locale.language
        choices = self.possible_languages()
        if lang in choices:
            return lang
        if 'en' in choices:
            return 'en'
        if 'ru' in choices:
            return 'ru'
        return choices[0]

    def chosen_country(self):
        return filter(None, [
            self._country,
            self._settings_country,
            self._gps_country,
        ] + self._sim_countries + [
            self._ip_country,
            self._locale.country,
            'RU'
        ])[0]

    def national_version(self):
        return {
            'RU': 'ru',
            'KZ': 'kz',
            'BY': 'ru',
            'UA': 'ua',
            'TR': 'tr',
        }.get(self.chosen_country(), 'com')


@api_v2.view(
    '/hello/',
    methods=['GET', 'POST'] if settings.DEBUG else ['POST'],
    argparser=LoggingRawArgParser(logging.getLogger('hello_v2'))
)
@skip_None_values
@api_v2.process_viewparams
def hello_v2(
    uuid=ViewParam(type_=ArgUUID()),
    oauth_token=ViewParam(required=False),
    appuser=ViewParam(name='app_key', type_=ArgAppUserByAppKey()),
    push_token=ViewParam(required=False),
    timestamp=ViewParam(type_=DateTimeGmt()),

    country=ViewParam(type_=IsoCountryArg(), required=False),
    language=ViewParam(type_=IsoLanguageArg(), required=False),
    currency=ViewParam(type_=IsoCurrencyArg(), required=False),

    settings_country=ViewParam(type_=IsoCountryArg(), required=False),
    locale=ViewParam(type_=IsoLocaleArg(), required=False),

    sim_countries=ViewParam(
        type_=travel.avia.avia_api.ant.custom_types.ArgsList(IsoCountryArg()), required=False
    ),
    gps_country=ViewParam(type_=IsoCountryArg(), required=False),

    known_languages=ViewParam(
        type_=travel.avia.avia_api.ant.custom_types.ArgsList(IsoLanguageArg())
    ),
    known_currencies=ViewParam(
        type_=travel.avia.avia_api.ant.custom_types.ArgsList(IsoCurrencyArg())
    )

):
    """ Hello_v2: Начало сессии """

    log.info(
        'Hello_v2 request: uuid=%r, oauth_token=%r, app_key=%s, '
        'push_token=%r, country=%r, language=%r, settings_country=%r, '
        'locale=%r, sim_countries=%r, gps_country=%r, appuser: %r',
        uuid, oauth_token, appuser.app_key,
        push_token, country, language, settings_country, locale,
        sim_countries, gps_country, appuser
    )

    if locale is None:
        # Костыль, проверяющий, не ходит ли к нам какое-нибудь приложение с
        # null-локалью. Если не ходят, стоит вернуть локали обязательность.
        log.critical('None locale by uuid=%r!', uuid)
        locale = IsoLocale(None, None)

    if 'de' in known_languages:
        log.warning('German is known: %s', known_languages)

    login_user(appuser)

    ip_city, _ = get_city_by_ip(real_client_ip(request))

    ip_country_code = None
    if ip_city and ip_city.country.code:
        try:
            ip_country_code = IsoCountryArg().clean(ip_city.country.code)
        except Exception as exc:
            log.critical(
                "Ip country detection returned a bad country [%r]: %r",
                ip_country_code, exc
            )

    guessed_settings = GuessedSettings(
        country=country,
        language=language,
        currency=currency,

        settings_country=settings_country,
        locale=locale,
        sim_countries=sim_countries,
        gps_country=gps_country,
        ip_country=ip_country_code,

        known_languages=known_languages,
        known_currencies=known_currencies,
    )

    new_session = hello(
        session=session,
        uuid=uuid,
        oauth_token=oauth_token,
        push_token=push_token,
        platform=appuser.platform,
        device_time=timestamp,
        lang=guessed_settings.chosen_language(),
        country=guessed_settings.chosen_country(),
        currency=guessed_settings.chosen_currency(),
        national_version=guessed_settings.national_version(),
        remote_address=real_client_ip(request),
        known_device_id=session.get('device_id')
    )

    session.clear()
    session.update(new_session)

    return OrderedDict({
        'ttl_seconds': appuser.session_ttl,
        'X-SESSION': session.get_xsession_value(),
        'oauth_token_invalid': (oauth_token and not session['yandex_uid']),
        'client_city': ip_city.point_key if ip_city else None,
        'languages': guessed_settings.possible_languages(),
        'currencies': guessed_settings.possible_currencies(),
        'language': guessed_settings.chosen_language(),
        'country': guessed_settings.chosen_country(),
        'currency': guessed_settings.chosen_currency(),
    })


if settings.DEBUG:
    @api.view('/hello/guest/')
    def guest():
        return {
            'appuser.id': current_user.get_id(),
            'session': OrderedDict(sorted(session.items())),
        }

    @api.view('/hello/appuser/')
    @login_required
    def get_appuser():
        u = current_user

        return {
            'appuser': repr(u),
            'appuser.id': u.get_id(),
            'device_time': u.device_time,
            'session': OrderedDict(sorted(session.items())),
            'user_created': u.created,
            'user_age': u.age.total_seconds(),
        }
