# -*- coding: utf-8 -*-
import logging

from flask_login import current_user, login_required

from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.avia.v1.model.user import User
from travel.avia.avia_api.avia.v1.model.favorite import MinPriceNotification, Favorite
from travel.avia.avia_api.avia.v1.views.base import api
from travel.avia.avia_api.avia.v1.schemas import fields, PassengersSchema, ApiSchema, TariffSchema

log = logging.getLogger(__name__)


class FavoriteSchema(ApiSchema):
    """ JSON объект https://st.yandex-team.ru/AVIAIOS-263 """

    key = fields.String()
    direct_only = fields.Boolean(default=False)
    point_from = fields.String(attribute='point_from_key', required=True)
    point_to = fields.String(attribute='point_to_key', required=True)
    date_forward = fields.DateYMD(required=True)
    date_backward = fields.DateYMD()
    service = fields.String()
    variants = fields.Integer()
    passengers = fields.Nested(PassengersSchema)
    min_price = fields.Nested(TariffSchema)
    filter = fields.Raw(attribute='filter_params', default={})

    class Meta:
        ordered = True


@api.view('/favorite/list/', sort_order=21.2)
@login_required
@api.result_schema(FavoriteSchema, many=True)
def favorite_list(
    key=ViewParam(required=False),
    schema_context=None
):
    """ Получить элементы избранного списком """

    user = User.get_by_appuser(current_user)

    if key:
        return filter(lambda f: f.key == key, user.favorites)

    return user.favorites


@api.view('/favorite/drop/', sort_order=21.3)
@login_required
@api.process_viewparams
def favorite_drop(key=ViewParam(required=True)):
    """ Удалить элемент из избранного """

    user = User.get_by_appuser(current_user)

    user.update(pull__favorites__key=key)

    return 'ok'


class FavoriteStoreAddViewSchema(ApiSchema):
    favorite_key = fields.String(attribute='key')


@api.view('/favorite/add/', sort_order=21.1)
@login_required
@api.result_schema(FavoriteStoreAddViewSchema)
def favorite_add(favorite=ViewParam(name='data', schema=FavoriteSchema),
                 schema_context=None):
    """ Добавить элемент в избранное """

    return _favorite_store(favorite)


@api.view('/favorite/store/', sort_order=21.1)
@login_required
@api.result_schema(FavoriteStoreAddViewSchema)
def favorite_store(favorite=ViewParam(name='data', schema=FavoriteSchema),
                   schema_context=None):
    """ Запомнить элемент в избранном """

    return _favorite_store(favorite)


def _favorite_store(favorite_data):
    user = User.get_by_appuser(current_user)

    log.info(
        'favorite_store yandex_uid: %r  uuid: %r',
        user.yandex_uid, user.uuid
    )

    key = favorite_data.pop('key', None)

    favorite = None

    if key:
        # заменяем по ключу
        favorite = next((f for f in user.favorites if f.key == key), None)

    if favorite is None:
        # заменяем по направлению и дате
        favorite = user.find_favorite(
            favorite_data['point_from_key'], favorite_data['point_to_key'],
            favorite_data['date_forward'], favorite_data.get('date_backward'),
            favorite_data.get('direct_only', False),
        )

    if favorite:
        key = favorite.key

        for k, v in favorite_data.items():
            setattr(favorite, k, getattr(Favorite, k).to_python(v))

        favorite.validate()

        user.update(pull__favorites__key=key)
        user.update(add_to_set__favorites=favorite)

        user.reload()

        favorite = next((f for f in user.favorites if f.key == key), None)

    else:
        favorite_data['last_min_price_notification'] = MinPriceNotification(
            price=favorite_data['min_price'],
            when=None
        )

        favorite = Favorite(
            key=Favorite.key.default(),
            **favorite_data
        )

        favorite.validate()

        user.update(add_to_set__favorites=favorite)

    return favorite
