# -*- coding: utf-8 -*-
import logging
from collections import Counter
from datetime import timedelta
from itertools import tee

from django.conf import settings
from flask_login import login_required, current_user

from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.ant.custom_types import ArgLang, Date, ArgHttpSearchId
from travel.avia.avia_api.avia.daemon_http_api import TicketDaemon
from travel.avia.avia_api.avia.lib.flight_info import SearchFlightInfo
from travel.avia.avia_api.avia.lib.jsend import ApiFail
from travel.avia.avia_api.avia.lib.variants_reference import HttpVariantsReference
from travel.avia.avia_api.avia.v1.model.flight import Flight
from travel.avia.avia_api.avia.v1.model.user import User
from travel.avia.avia_api.avia.v1.schemas import (
    FlightListViewSchema, FlightFromSearchSchema,
    FlightAddFromSearchResultSchema, FlightAddListViewSchema, RawFlightSchema
)
from travel.avia.avia_api.avia.v1.views.base import api
from travel.avia.avia_api.avia.lib.user_flights import add_flight_manually, UnknownFlightException

log = logging.getLogger(__name__)


@api.view('/flight/add/manually/')
@login_required
@api.result_schema(FlightListViewSchema)
def flight_add_manually_handler(
    number=ViewParam(required=True),
    departure_date=ViewParam(type_=Date(), required=True),
    lang=ViewParam(type_=ArgLang(), required=False),
    schema_context=None
):
    user = User.get_by_appuser(current_user)
    schema_context['lang'] = lang or current_user.lang

    try:
        return add_flight_manually(user, number, departure_date)
    except UnknownFlightException as exc:
        raise ApiFail({
            'unknown_flights': [{
                'number': exc.number,
                'departure_datetime': exc.departure_date.strftime('%Y-%m-%d %H:%M')
            }]
        })


@api.view('/flight/add/from_search/')
@login_required
@api.result_schema(FlightAddFromSearchResultSchema)
def flight_add_from_search_handler(
    search_id=ViewParam(type_=ArgHttpSearchId(), required=True),
    raw_flights=ViewParam(schema=(FlightFromSearchSchema, dict(many=True))),
    lang=ViewParam(type_=ArgLang(), required=False),
    schema_context=None
):
    user = User.get_by_appuser(current_user)
    schema_context['lang'] = lang or current_user.lang
    return _add_flight_from_search(user, search_id, raw_flights)


def _add_flight_from_search(user, search_id, raw_flights):
    flights = []
    unknown_flights = []
    flights_to_notify_about = set()

    segments = HttpVariantsReference(
        TicketDaemon(settings.DAEMON_URL).search_results(
            qid=search_id.qid(),
            lang=current_user.lang,
            currency=current_user.currency,
        ).variants
    ).segments()

    for index, raw_flight in enumerate(raw_flights):
        try:
            search_info = SearchFlightInfo.from_segment(
                segment=next(
                    segment
                    for segment in segments
                    if (
                        segment.number == raw_flight['number'] and
                        segment.departure_station.point_key == raw_flight['departure'] and
                        segment.departs_at.date() == raw_flight['departure_datetime'].date()
                    )
                )
            )

            flight = Flight.get_or_create(
                number=search_info.number,
                departure_date=search_info.scheduled_departure_date(),
                departure=search_info.departure_point.point_key,
            ).updated_with_info(
                search_info
            )

            flights.append(flight)
            if index == 0:
                flights_to_notify_about.add(flight)

        except Exception as exc:
            log.exception(
                'Got an error while getting a flight from search: %r', exc
            )
            unknown_flights.append(raw_flight)

    if not unknown_flights:
        flights_to_notify_about.update(
            extract_flights_to_notify_aeroex(flights)
        )

    log.info(
        'Flights: %r. Unknown flights: %r. For notification: %r',
        flights, unknown_flights, flights_to_notify_about
    )

    if flights:
        user.update(pull_all__flights=flights)
        user.update(add_to_set__flights=flights)
        user.update(pull_all__aeroex_notify_flights=flights_to_notify_about)
        user.update(add_to_set__aeroex_notify_flights=flights_to_notify_about)
        user.reload()

        for f in flights:
            if f in user.aeroex_notify_flights:
                f.user_aeroexpress = f.aeroexpress

    return {
        'flights': set(flights),
        'unknown_flights': unknown_flights
    }


@api.view('/flight/add/', sort_order=19)
@login_required
@api.result_schema(FlightAddListViewSchema)
def flight_add(
    raw_flights=ViewParam(
        name='flights', schema=(RawFlightSchema, dict(many=True))
    ),
    lang=ViewParam(type_=ArgLang(), required=False),
    schema_context=None
):
    """ Добавить набор вылетов """

    user = User.get_by_appuser(current_user)
    schema_context['lang'] = lang or current_user.lang
    log.debug('Flight add: %r', raw_flights)

    if raw_flights and 'search_id' in raw_flights[0]:
        search_id = raw_flights[0]['search_id']
        return _add_flight_from_search(user, search_id, raw_flights)

    if len(raw_flights) != 1:
        log.warning(
            'User %r adds multiple flights without a search id: %r',
            user, raw_flights
        )

    if not raw_flights:
        return {'flights': []}

    return {
        'flights': [
            added_flight
            for raw_flight in raw_flights
            for added_flight in add_flight_manually(
                user,
                number=raw_flight['number'],
                departure_date=raw_flight['departure_datetime'].replace(
                    hour=0, minute=0, second=0, microsecond=0, tzinfo=None
                )
            )['flights']
        ]
    }


MIN_HOURS_TO_NOTIFY_ABOUT_AEROEXPRESS = 12


def pairwise(iterable):
    "s -> (s0,s1), (s1,s2), (s2, s3), ..."
    a, b = tee(iterable)
    next(b, None)
    return zip(a, b)


def extract_flights_to_notify_aeroex(flights):
    flights = sorted(flights, key=lambda f: f.utc_departure_dt)
    points = [f.departure for f in flights] + [flights[-1].arrival]
    common_points = {k for k, v in Counter(points).iteritems() if v > 1}
    pauses_n_flights = [
        (following.utc_departure_dt - prev.utc_arrival_dt, following)
        for prev, following in pairwise(flights)
    ]
    aeroexpress_candidates = [
        flight
        for pause, flight in sorted(pauses_n_flights, reverse=True)
        if pause > timedelta(hours=MIN_HOURS_TO_NOTIFY_ABOUT_AEROEXPRESS) and
        flight.departure not in common_points
    ]

    return aeroexpress_candidates[:1]
