# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

from django.conf import settings
from django.db.models import Q
from flask import make_response, request
from flask_login import login_required

from travel.avia.library.python.common.models.geo import (
    CityMajority, Country, Settlement, Station2Settlement, StationType,
    StationCode, Station, CodeSystem,
)
from travel.avia.library.python.common.models.translations import TranslatedTitle
from travel.avia.library.python.common.models_utils.geo import Point
from travel.avia.library.python.common.settings import LANGUAGE_FALLBACKS

from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.avia.lib.cache import JsonFileCache
from travel.avia.avia_api.avia.lib.date import (
    format_aware_dt_for_browser, format_aware_dt_utc, parse_rfc2822_datetime
)
from travel.avia.avia_api.avia.lib.decorators import elementwise, skip_None_values
from travel.avia.avia_api.avia.lib.jsend import ApiFail, JsendSuccess
from travel.avia.avia_api.ant.custom_types import ArgLang
from travel.avia.avia_api.avia.v1.views.base import api

log = logging.getLogger(__name__)


def get_if_modified_since_header():
    raw_header = request.headers.get('If-Modified-Since')

    if not raw_header:
        return None

    return parse_rfc2822_datetime(raw_header)


def get_por(lang):
    por_cache = JsonFileCache(settings.POR_CACHE_FILENAME[lang])

    if not por_cache.exists():
        log.warning(
            "Por cache for [%s] doesn't exist; creating a new one", lang
        )
        data = serialize_por(collect_raw_por(), lang)
        if not por_cache.exists():  # double-check
            por_cache.save(data)

    last_modified_at = por_cache.last_modified_at
    if_modified_since = get_if_modified_since_header()

    log.info(
        'Lang: %s, If-Modified-Since: %r, Last-Modified: %r',
        lang,
        format_aware_dt_utc(if_modified_since),
        format_aware_dt_utc(last_modified_at),
    )
    if if_modified_since and last_modified_at:
        if last_modified_at <= if_modified_since:
            return make_response('', 304)

    log.info('Por full-response')

    r = JsendSuccess(data=por_cache.load()).get_response()

    r.headers['Last-Modified'] = (
        format_aware_dt_for_browser(por_cache.last_modified_at)
    )

    return r


@api.view('/info/por/')
@login_required
@api.process_viewparams
def por(lang=ViewParam(type_=ArgLang())):
    return get_por(lang)


def collect_raw_por():
    station_codes = list(
        StationCode.objects
            .filter(system__code__in=['iata', 'sirena'])
            .namedtuples('system_id', 'station_id', 'code')
    )

    code_systems = {
        s.id: s.code
        for s in CodeSystem.objects.all()
    }

    stations = list(
        Station.objects
            .filter(id__in={s.station_id for s in station_codes})
            .namedtuples(
                'id', 'settlement_id', 'country_id', 'has_aeroexpress',
                'station_type_id', 'title', 'new_L_title_id',
            )
    )

    iata_codes = {
        code.station_id: code.code
        for code in station_codes
        if code_systems.get(code.system_id) == 'iata'
    }

    sirena_codes = {
        code.station_id: code.code
        for code in station_codes
        if code_systems.get(code.system_id) == 'sirena'
    }

    relational_settlement_ids = set(
        Station2Settlement.objects
            .filter(station__in={s.id for s in stations})
            .values_list('settlement_id', flat=True)
    )

    settlements = list(
        Settlement.objects
            .filter(
                Q(id__in={s.settlement_id for s in stations}) |
                Q(id__in=relational_settlement_ids)
            )
            .namedtuples(
                'id', '_geo_id', 'country_id', 'iata', 'sirena_id',
                'majority_id', 'title', 'new_L_title_id',
            )
    )

    city_majorities = list(
        CityMajority.objects
            .filter(id__in=set(s.majority_id for s in settlements))
            .namedtuples()
    )

    station_types = list(
        StationType.objects
            .filter(id__in=set(s.station_type_id for s in stations))
            .namedtuples()
    )

    related_countries = list(
        Country.objects
            .filter(
                Q(id__in={s.country_id for s in stations}) |
                Q(id__in={c.country_id for c in settlements})
        )
        .order_by('code')
        .namedtuples('id', 'code', '_geo_id', 'title', 'new_L_title_id')
    )

    titles = {
        t.id: t
        for t in TranslatedTitle.objects
        .filter(
            id__in={
                c.new_L_title_id for c in related_countries
            } | {
                c.new_L_title_id for c in settlements
            } | {
                s.new_L_title_id for s in stations
            } | {
                t.new_L_name_id for t in station_types
            }
        )
        .namedtuples('id', *['{}_nominative'.format(l) for l in settings.AVIA_API_LANGUAGES])
    }

    return {
        'countries': related_countries,
        'cities': settlements,
        'airports': stations,
        'iata_codes': iata_codes,
        'sirena_codes': sirena_codes,
        'city_majority': city_majorities,
        'station_type': station_types,
        'titles': titles,
    }


def translate(title, lang):
    if title:
        def gen():
            yield getattr(title, '{}_nominative'.format(lang), None)
            for fallback_lang in LANGUAGE_FALLBACKS.get(lang, ['en']):
                yield getattr(title, '{}_nominative'.format(fallback_lang), None)
            yield getattr(title, 'ru_nominative')

        for t in gen():
            if t:
                return t


def serialize_por(raw_por, lang):
    return {
        'countries': country_json(raw_por['countries'], raw_por['titles'], lang=lang),
        'cities': settlement_json(raw_por['cities'], raw_por['titles'], lang=lang),
        'airports': stations_json(
            raw_por['airports'], raw_por['iata_codes'], raw_por['sirena_codes'],
            raw_por['titles'], lang=lang
        ),
        'city_majority': city_majority_json(raw_por['city_majority']),
        'station_type': station_type_json(raw_por['station_type'], raw_por['titles'], lang=lang),
    }


@elementwise
@skip_None_values
def country_json(c, titles, lang):
    return {
        'key': Point.get_point_key(Country, c.id),
        'title': translate(titles.get(c.new_L_title_id), lang) or c.title,
        'geoid': c.geo_id,
        'iso_code': c.code,
    }


@elementwise
@skip_None_values
def settlement_json(s, titles, lang):
    return {
        'key': Point.get_point_key(Settlement, s.id),
        'geoid': s.geo_id,
        'country_key': Point.get_point_key(Country, s.country_id) if s.country_id else None,
        'is_capital': (s.majority_id == CityMajority.CAPITAL_ID),
        'majority_id': s.majority_id,
        'title': translate(titles.get(s.new_L_title_id), lang) or s.title,
        'iata': s.iata,  # Атрибут из базы
        'sirena': s.sirena_id,  # Атрибут из базы
    }


def stations_json(stations, iata_codes, sirena_codes, titles, lang):
    return [
        station_json(station, iata_codes, sirena_codes, titles, lang)
        for station in stations
    ]


@skip_None_values
def station_json(station, iata_codes, sirena_codes, titles, lang):
    return {
        'key': Point.get_point_key(Station, station.id),
        'city_key': Point.get_point_key(Settlement, station.settlement_id) if station.settlement_id else None,
        'title': translate(titles.get(station.new_L_title_id), lang) or station.title,
        'iata': iata_codes.get(station.id),  # Значение из StationCode
        'sirena': sirena_codes.get(station.id),  # Значение из StationCode
        'has_aeroexpress': station.has_aeroexpress,
        'station_type_id': station.station_type_id,
    }


@elementwise
@skip_None_values
def station_type_json(st, titles, lang):
    return {
        'id': st.id,
        'title': translate(titles.get(st.new_L_name_id), lang),
    }


@elementwise
@skip_None_values
def city_majority_json(m):
    return {
        'id': m.id,
        'title': m.title,
    }


@api.view('/fail/', make_spec=False)
def fail():
    raise ApiFail('FAIL')
