# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

from django.conf import settings
from flask import request
from flask_login import current_user, login_required

from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.ant.custom_types import ArgLang, ArgTags, ArgHttpSearchId, ArgHttpChance
from travel.avia.avia_api.avia.daemon_http_api import TicketDaemon
from travel.avia.avia_api.avia.lib.jsend import ApiFail
from travel.avia.avia_api.avia.lib.money_logs import write_show_log, log_redirect
from travel.avia.avia_api.avia.lib.remote_addr import real_client_ip
from travel.avia.avia_api.avia.lib.variants_reference import (
    HttpVariantsReference, inject_fictional_companies
)
from travel.avia.avia_api.avia.v1.schemas.order import RedirectDataSchema, HttpOrderViewSchema
from travel.avia.avia_api.avia.v1.views.base import api

log = logging.getLogger(__name__)


@api.view('/order/')
@login_required
@api.result_schema(HttpOrderViewSchema)
def order_http_handler(
    search_id=ViewParam(type_=ArgHttpSearchId(), required=True),
    tags=ViewParam(name='vtags', type_=ArgTags()),
    lang=ViewParam(type_=ArgLang()),
    schema_context=None
):
    if search_id.national_version() != current_user.national_version:
        log.error(
            'Different national versions in order. query: %s user: %s',
            search_id.national_version(), current_user.national_version
        )

    variants = TicketDaemon(settings.DAEMON_URL).search_results(
        qid=search_id.qid(),
        lang=lang or current_user.lang,
        currency=current_user.currency,
        tags=tags,
    ).variants

    inject_fictional_companies(variants)

    reference = HttpVariantsReference(variants)

    write_show_log(
        variants,
        national_version=search_id.national_version(),
        point_from=search_id.point_from(),
        point_to=search_id.point_to(),
        user_ip=real_client_ip(request),
        platform=settings.BILLING_PLATFORMS.get(current_user.platform),
        yandexuid=current_user.yandex_uid or current_user.uuid,
    )

    schema_context.update({
        'national_version': search_id.national_version(),
        'lang': lang,
        'qid': search_id.qid(),
    })

    return {
        'variants': reference.variants(),
        'statuses': {
            partner.code: 'done'
            for partner in reference.partners()
        },
        # Если в выдачу приложения попадает пустой трип, происходит креш
        'trips': [trip for trip in reference.trips() if trip.segments],
        'companies': reference.companies(),
        'partners': reference.partners(),
    }


@api.view('/order/redirect/')
@login_required
@api.result_schema(RedirectDataSchema)
def order_redirect_http_handler(
    chance=ViewParam(type_=ArgHttpChance()),
    schema_context=None
):
    try:
        redirect_data = TicketDaemon(settings.DAEMON_URL).redirect_data(
            qid=chance.qid(),
            order_data=chance.order_data,
            yandex_uid=current_user.yandex_uid or current_user.uuid,
            passport_uid=current_user.yandex_uid,
            user_ip=real_client_ip(request),
            lang=chance.lang or current_user.lang,
        )
        log_redirect(
            chance, chance.partner(),
            redirect_url=redirect_data['url'],
            platform=settings.BILLING_PLATFORMS.get(current_user.platform),
            user_ip=real_client_ip(request),
            passportuid=current_user.yandex_uid,
            yandexuid=current_user.yandex_uid or current_user.uuid,
        )
    except Exception as exc:
        log.exception(u'Order redirection error %r', exc)
        raise ApiFail('Redirection error')

    return {
        'url': redirect_data['url'],
        'post': redirect_data['post']
    }
