# -*- coding: utf-8 -*-
import logging

from django.conf import settings
from flask_login import login_required, current_user

from travel.avia.avia_api.ant.api_interface import ViewParam
from travel.avia.avia_api.ant.custom_types import (
    ArgLang, ArgSearchPages, ArgCSPartnersCodes, ArgHttpSearchId
)
from travel.avia.avia_api.avia.daemon_http_api import TicketDaemon
from travel.avia.avia_api.avia.lib.variants_reference import (
    HttpVariantsReference, inject_fictional_companies,
)
from travel.avia.avia_api.avia.v1.schemas import HttpSearchResultsViewSchema
from travel.avia.avia_api.avia.v1.views.base import api

log = logging.getLogger(__name__)


@api.view('/search/results/')
@login_required
@api.result_schema(HttpSearchResultsViewSchema)
def mobile_search_results_handler(
    search_id=ViewParam(type_=ArgHttpSearchId(), name='search_id'),
    lang=ViewParam(type_=ArgLang()),
    skip_partners=ViewParam(
        type_=ArgCSPartnersCodes(), required=False, default=None
    ),
    pages=ViewParam(type_=ArgSearchPages(), required=False, default=None),
    schema_context=None
):
    if search_id.national_version() != current_user.national_version:
        log.error(
            'Different national versions. query: %s user: %s',
            search_id.national_version(), current_user.national_version
        )

    if pages:
        log.critical('The pages parameter is used: %r', pages)

    schema_context.update({
        'national_version': current_user.national_version,
        'lang': lang,
    })

    result = TicketDaemon(settings.DAEMON_URL).search_results(
        qid=search_id.qid(),
        lang=lang or current_user.lang,
        currency=current_user.currency,
        skip_partners=skip_partners,
        result_pages=pages,
    )

    variants = result.variants
    inject_fictional_companies(variants)

    reference = HttpVariantsReference(variants)

    # RASPTICKETS-4324 Ограничить минимальный ttl предложений
    for variant in variants:
        variant.ttl += 300

    return {
        'variants': variants,
        'statuses': {
            partner_code: 'done' if status in {'empty', 'done'} else 'querying'
            for partner_code, status in result.statuses.items()
        },
        # Если в выдачу приложения попадает пустой трип, происходит креш
        'trips': [trip for trip in reference.trips() if trip.segments],
        'companies': reference.companies(),
        'partners': reference.partners(),
        'search_id': search_id.encrypt(),
    }
