import imp
import logging

import os
import re
import uuid

from django.conf import settings
from mongolock import MongoLock

from travel.avia.avia_api.avia.v1.model.db import db

log = logging.getLogger(__name__)
log.addHandler(logging.StreamHandler())


class ExecutedMigration(db.Document):
    meta = {'collection': 'migrations'}

    name = db.StringField(required=True, unique=True)


class MigrationFail(Exception):
    pass


class MigrationModule(object):
    MIGRATIONS_FILENAME_REGEX = re.compile(r'm(?P<number>[0-9]+)\.py')

    def __init__(self, migration_filename):
        self.filename = migration_filename

    @property
    def migration(self):
        module_name = re.sub(r'\.py$', '', self.filename)

        module = imp.load_source(
            module_name,
            os.path.join(
                settings.MIGRATIONS_PATH, self.filename
            )
        )

        return module.Migration()

    @property
    def number(self):
        match = self.MIGRATIONS_FILENAME_REGEX.match(self.filename)
        if not match:
            return None
        return int(match.group('number'))


class PendingMigrations(object):
    def __init__(self):
        self.lock = MongoLock(
            collection=db.connection[
                settings.MONGO_DATABASE
            ][
                settings.MIGRATION_LOCKS_COLLECTION
            ]
        )
        self.id = str(uuid.uuid4())

    def execute(self, spec_migration=None):
        with self.lock('mongo_migration', self.id, timeout=None, expire=600):
            modules = self._get_pending_migration_modules(spec_migration)

            if not modules:
                log.info('There are no new migrations')
                return

            for module in modules:
                log.info(u'Executing migration %s...', module.filename)

                migration = module.migration
                try:
                    migration.execute()
                except Exception as exc:
                    log.exception(
                        u'Error while executing migration. Rollback. %r', exc
                    )
                    migration.rollback()
                    raise

                ExecutedMigration(name=module.filename).save()

    def rollback(self, spec_migration):
        with self.lock('mongo_migration', self.id, timeout=None, expire=600):
            [module] = self._get_pending_migration_modules(spec_migration)
            log.info(u'Rollbacking migration %s...', spec_migration)
            module.migration.rollback()

    def _get_pending_migration_modules(self, spec_migration=None):
        if not os.path.exists(settings.MIGRATIONS_PATH):
            log.error(
                u'Migrations are not found at %r', settings.MIGRATIONS_PATH
            )
            raise MigrationFail(u'Migrations are not found')

        migration_filenames = [
            fname
            for fname in os.listdir(settings.MIGRATIONS_PATH)
            if (fname.endswith('.py') and
                fname != '__init__.py' and
                not ExecutedMigration.objects(name=fname))
        ]

        if spec_migration is not None:
            if spec_migration not in migration_filenames:
                log.error(
                    u'Migration %r is not found. Available names: %r',
                    spec_migration, migration_filenames
                )
                raise MigrationFail(u'Migration is not found')

            migration_filenames = [spec_migration]

        modules = [MigrationModule(fname) for fname in migration_filenames]

        for module in modules:
            if module.number is None:
                log.error(
                    u'%r is not a valid migration filename', module.filename
                )
                raise MigrationFail(u"Migration name isn't found")

        modules.sort(key=lambda m: m.number)
        return modules
