# -*- coding: utf-8 -*-
"""
Usage:
    $./manage.py run_path scripts/push_min_price.py --help
"""
from __future__ import unicode_literals

import json
import logging
import sys

import click
from click import echo

from avia.lib.push_notifications import MinPricePushNotification
from avia.v1.model.user import User
from avia.v1.model.device import Device
from avia.v1.views.favorites import FavoriteSchema

log = logging.getLogger('')
log.addHandler(logging.StreamHandler(sys.stdout))
log.setLevel(logging.INFO)


class UserByUUID(click.ParamType):
    name = 'uuid_user'

    def convert(self, value, param, ctx):
        try:
            return User.objects.get(uuid=value)
        except User.DoesNotExist as e:
            try:
                device = Device.objects.get(uuid=value)
            except Device.DoesNotExist as device_ex:
                log.info(
                    'No device with uuid [%s]: %s',
                    value, device_ex
                )
            else:
                return device.user

            self.fail('No user with uuid [%s]: %s' % (value, e), param, ctx)


class UserByYandexUID(click.ParamType):
    name = 'yandex_uid_user'

    def convert(self, value, param, ctx):
        try:
            return User.objects.get(yandex_uid=value)
        except User.DoesNotExist as e:
            self.fail(
                'No user with yandex_uid [%s]: %s' % (value, e), param, ctx
            )


@click.command()
@click.option('-v', '--verbose', is_flag=True)
@click.option('--uuid', type=UserByUUID())
@click.option('--yandex-uid', type=UserByYandexUID())
@click.option('--favorite-key')
def main(verbose, uuid, yandex_uid, favorite_key):
    if verbose:
        log.setLevel(logging.DEBUG)

    user = uuid or yandex_uid

    if not user:
        echo('No user to push')
        return

    try:
        favorite = {f.key: f for f in user.favorites}[favorite_key]
    except KeyError:
        show_user_favorites(user)
    else:
        MinPricePushNotification(favorite).send([user])


def show_user_favorites(user):
    echo('User favorites:')
    favo_schema = FavoriteSchema(strict=True)

    for f in user.favorites:
        echo(json.dumps(favo_schema.dump(f).data, indent=4))


if __name__ == '__main__':
    main()
