# -*- coding: utf-8 -*-
"""
Usage:
    $./manage.py run_path scripts/push_scenario.py --help
"""
from __future__ import unicode_literals

import json
import logging
import sys

import click
from click import echo
from django.conf import settings

from avia.lib import pushes
from avia.v1.model.user import User


log = logging.getLogger('')
log.addHandler(logging.StreamHandler(sys.stdout))
log.setLevel(logging.INFO)


class UserByYandexUID(click.ParamType):
    name = 'yandex_uid_user'

    def convert(self, value, param, ctx):
        try:
            return User.objects.get(yandex_uid=value)
        except User.DoesNotExist as e:
            self.fail(
                'No user with yandex_uid [%s]: %s' % (value, e), param, ctx
            )


class JsonData(click.ParamType):
    name = 'jsondata'

    def convert(self, value, param, ctx):
        try:
            return json.loads(value)
        except KeyError as e:
            self.fail(str(e), param, ctx)


class Platform(click.ParamType):
    name = 'platform'

    def convert(self, value, param, ctx):
        try:
            settings.PUSHER_DATAS[value]
            return value
        except KeyError as e:
            self.fail(
                '%s. Available platforms: %r' % (
                    e, settings.PUSHER_DATAS.keys()
                ),
                param, ctx
            )


@click.command()
@click.option('-v', '--verbose', is_flag=True)
@click.option('--uuid', required=True)
@click.option('--yandex-uid', type=UserByYandexUID())
@click.option('--push-token', required=True)
@click.option('--platform', required=True, type=Platform())
@click.option('--data', type=JsonData())
@click.option('--message', required=True)
def main(verbose, uuid, yandex_uid, push_token, platform, data, message):
    yandex_uid_user = yandex_uid

    if verbose:
        log.setLevel(logging.DEBUG)

    try:
        user = User.objects.get(uuid=uuid)
    except User.DoesNotExist:
        user = None
        log.debug('No user with uuid [%s]', uuid)

    users = filter(None, [user, yandex_uid_user])
    if not users:
        echo('No users to push')
        return

    yandex_uid = yandex_uid_user and yandex_uid_user.yandex_uid or None
    pushes.v2_add(push_token, yandex_uid, uuid, platform)


if __name__ == '__main__':
    main()
