# -*- coding: utf-8 -*-
import logging
from datetime import datetime, date
from argparse import ArgumentParser

import ydb
from sandbox.projects.Travel.resources import dicts
from sandbox.projects.avia_resources.resources import AVIA_WIZARD_DIRECTION_NATIONAL
from sandbox.common.rest import Client as SandboxClient
from sandbox.common.proxy import OAuth

from travel.avia.library.python.lib_yt.client import configured_client
from travel.library.python.dicts.avia.direction_national_repository import DirectionNationalRepository
from travel.library.python.dicts.avia.settlement_repository import SettlementRepository
from travel.library.python.dicts.avia.near_direction_repository import NearDirectionRepository
from travel.avia.avia_statistics.alternative_routes_prices_updater.lib.updater import AlternativeRoutesPricesUpdater
from travel.avia.avia_statistics.alternative_routes_prices_updater.lib.table import AlternativeRoutesPriceTable
from travel.avia.avia_statistics.lib.settlements_geo_index import SettlementsGeoIndex
from travel.avia.avia_statistics.services.price_index.client import Client as PriceIndexClient
from travel.avia.avia_statistics.landing_routes import get_landing_routes
from travel.avia.library.python.ydb.session_manager import YdbSessionManager
from travel.avia.library.python.sandbox.resource_fetcher import ResourceFetcher

logger = logging.getLogger(__name__)


def main():
    parser = ArgumentParser()

    parser.add_argument('--debug', action='store_true', default=False)
    parser.add_argument('--price-index-host', help='price-index host',
                        default='http://price-index.production.avia.yandex.net')

    parser.add_argument('--yt-token', help='YT token', required=True)
    parser.add_argument('--yt-proxy', default='hahn')

    parser.add_argument('--ydb-cluster', default='ydb-ru-prestable.yandex.net:2135')
    parser.add_argument('--ydb-database', default='/ru-prestable/ticket/testing/avia_statistics')
    parser.add_argument('--ydb-table', default='alternative_routes_prices')
    parser.add_argument('--ydb-token', help='YDB token', required=True)

    parser.add_argument('--sandbox-token', help='Sandbox token', required=True)

    parser.add_argument('--batch', help='Batch size', default=100)
    parser.add_argument('--window-size', help='Price searching window size in the future', default=30)
    parser.add_argument('--date', help='Date', default=date.today().isoformat())

    arguments = parser.parse_args()

    logging.basicConfig(level=logging.INFO)

    if arguments.debug:
        logging.basicConfig(level=logging.DEBUG)

    sandbox_oauth = OAuth(arguments.sandbox_token)
    sandbox_client = SandboxClient(auth=sandbox_oauth)
    resource_fetcher = ResourceFetcher(sandbox_client, sandbox_oauth)

    settlement_repository = SettlementRepository()
    settlement_resource_data = resource_fetcher.fetch_latest_ready(dicts.TRAVEL_DICT_AVIA_SETTLEMENTS_PROD)
    settlement_repository.load_from_string(settlement_resource_data)

    direction_national_repository = DirectionNationalRepository()
    direction_national_resource_data = resource_fetcher.fetch_latest_ready(AVIA_WIZARD_DIRECTION_NATIONAL)
    direction_national_repository.load_from_string(direction_national_resource_data)

    near_direction_repository = NearDirectionRepository()
    near_direction_resource_data = resource_fetcher.fetch_latest_ready(dicts.TRAVEL_DICT_AVIA_NEAR_DIRECTION_PROD)
    near_direction_repository.load_from_string(near_direction_resource_data)

    settlements_geo_index = SettlementsGeoIndex(direction_national_repository, settlement_repository)

    yt_client = configured_client(proxy=arguments.yt_proxy, token=arguments.yt_token)
    ydb_config = ydb.DriverConfig(
        endpoint=arguments.ydb_cluster,
        database=arguments.ydb_database,
        auth_token=arguments.ydb_token,
    )
    session_manager = YdbSessionManager(ydb_config)

    alternative_routes_price_table = AlternativeRoutesPriceTable(session_manager, arguments.ydb_database, arguments.ydb_table)
    price_index_client = PriceIndexClient(arguments.price_index_host)
    landing_routes = get_landing_routes(yt_client)
    alternative_routes_prices_updater = AlternativeRoutesPricesUpdater(
        landing_routes,
        near_direction_repository,
        settlements_geo_index,
        alternative_routes_price_table,
        price_index_client,
        arguments.batch,
    )
    target_date = datetime.strptime(arguments.date, '%Y-%m-%d')
    alternative_routes_prices_updater.update(target_date, arguments.window_size)


if __name__ == '__main__':
    main()
