package main

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/cachedtvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/app"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/countrysearch"
	countrysearchhandler "a.yandex-team.ru/travel/avia/avia_statistics/api/internal/countrysearch/handler"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/landingcityto"
	landingcitytohandler "a.yandex-team.ru/travel/avia/avia_statistics/api/internal/landingcityto/handler"
	landingrouteshandler "a.yandex-team.ru/travel/avia/avia_statistics/api/internal/landingroutes/handler"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/pkg/dicts"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/pkg/repositories"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/pkg/tables"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/pkg/ydb"
	seolandinghandler "a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/handler"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/service/landings"
	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/library/go/configuration"
	dynamicresources "a.yandex-team.ru/travel/library/go/dicts/updaterservice"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/syncutil"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
)

func main() {
	maxprocs.AdjustAuto()

	ctx, ctxCancel := context.WithCancel(context.Background())
	defer ctxCancel()
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &app.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		return
	}

	logger, err := logging.New(&app.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger:", err)
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer("avia-statistics")
	logger.Info("jaeger tracer was initialized")

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			logger.Error("failed to close logger", log.Error(err))
			return
		}
	}()

	dictsRegistry := dicts.NewRegistry()
	runDynamicResourcesService(logger, dictsRegistry.GetUpdatablesByPaths())

	var tvmClient tvm.Client
	tvmAllowedIds := tvmutil.TvmClientIDFromInt(app.Cfg.Tvm.WhiteList)
	if app.Cfg.Tvm.Enabled {
		tvmClient, err = tvmtool.NewDeployClient()
		if err != nil {
			logger.Error("failed to create tvm client", log.Error(err))
			return
		}
		tvmClient, err = cachedtvm.NewClient(
			tvmClient,
			cachedtvm.WithCheckServiceTicket(
				app.Cfg.Tvm.CacheTTL,
				app.Cfg.Tvm.CacheMaxItems,
			),
		)
		if err != nil {
			logger.Fatal("failed to create cached tvm client", log.Error(err))
		}
	}

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()

	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, app.Cfg.Metrics.PerfMetricsRefreshInterval)

	sessionPool, err := ydb.NewSessionPool(context.Background(), &app.Cfg.Ydb)
	if err != nil {
		logger.Error("failed to create YDB session pool", log.Error(err))
		return
	}
	logger.Info("YDB session pool was created")
	tablesRegistry := tables.NewRegistry(sessionPool, tables.DefaultConfig)
	repositoriesRegistry, err := repositories.NewRegistry(repositories.DefaultConfig, tablesRegistry, logger)
	if err != nil {
		logger.Error("failed to create repository", log.Error(err))
		return
	}
	logger.Info("repositories were created")

	landingService := landings.NewService(
		logger,
		repositoriesRegistry.AlternativeRoutesPrices.Get,
		repositoriesRegistry.MedianPrices.Get,
		repositoriesRegistry.PopularMonthsRepository.Get,
		repositoriesRegistry.ReturnTicketPricesRepository.Get,
		repositoriesRegistry.RouteInfos.Get,
		repositoriesRegistry.TopAirlinesRepository.Get,
		repositoriesRegistry.MinPricesByAirline.Get,
		repositoriesRegistry.RouteCrosslinks.Get,
		repositoriesRegistry.Routes.Contains,
	)
	logger.Info("landing service was created")

	landingCityToService := landingcityto.NewCityToService(
		logger,
		repositoriesRegistry.CityToRouteCrosslinks.Get,
		repositoriesRegistry.CityToMonthAndYearPrices.Get,
		repositoriesRegistry.CityToNearestCities.Get,
	)
	logger.Info("landing city to service was created")

	httpLandingHandler := seolandinghandler.NewHTTPLandingHandler(landingService, logger)
	landingCityToHandler := landingcitytohandler.NewHTTPLandingCityToHandler(landingCityToService, logger)
	landingRoutesHandler := landingrouteshandler.NewHTTPLandingRoutesHandler(repositoriesRegistry.Routes, logger)

	cityListGetter := countrysearch.NewCityListGetter(
		dictsRegistry.StationRepository,
		dictsRegistry.SettlementRepository,
		repositoriesRegistry.SettlementPopularity,
		dictsRegistry.StationCodesRepository,
		logger,
	)
	countrySearchHandler := countrysearchhandler.NewCountrySearchHandler(dictsRegistry, cityListGetter, logger)

	httpRouteBuilders := []httpserver.RouteBuilder{
		httpLandingHandler.GetRouteBuilder(),
		landingCityToHandler.GetRouteBuilder(),
		landingRoutesHandler.GetRouteBuilder(),
		countrySearchHandler.GetRouteBuilder(),
	}

	// add readiness and shutdown hooks
	probeState := probes.NewState(logger, probes.OnStopAfter(ctxCancel))
	probeRouteBuilders := probes.GetChiRouteBuilders(&app.Cfg.Probes, probeState)
	httpRouteBuilders = append(httpRouteBuilders, probeRouteBuilders...)

	wg := syncutil.WaitGroup{}
	wg.Go(func() {
		logger.Info("Started HTTP server")
		err = httpserver.RunHTTPServer(context.Background(), app.Cfg.HTTP, httpRouteBuilders, logger, tvmClient, tvmAllowedIds, rootRegistry.WithPrefix("http"))
		if err != nil {
			logger.Fatal("Error while starting HTTP server", log.Error(err))
		}
	})

	wg.Go(func() {
		logger.Info("Started Metrics server")
		err = metricserver.RunMetricsHTTPServer(context.Background(), app.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	})
	wg.Wait()
}

func runDynamicResourcesService(logger log.Logger, updatableByPaths map[string]dynamicresources.Updatable) {
	registry := dynamicresources.NewRegistry(app.Cfg.Dicts.ResourcesPath, app.Cfg.Dicts.UseDynamicResources)
	registry.RegisterBatch(updatableByPaths)
	err := registry.OnUpdate()
	if err != nil {
		logger.Fatal("Error while initializing dicts", log.Error(err))
	}
	logger.Info("dicts were created")

	dynamicResourceService := dynamicresources.NewService(
		logger,
		app.Cfg.DynamicResources,
		registry.OnUpdate,
	)
	dynamicResourceService.BackroundRun()
}
