package countrysearch

import (
	"context"
	"sort"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/pkg/dicts"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/pkg/repositories"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

const CityTopNumber = 100

type CityListGetter struct {
	stationRepository    *dicts.StationRepository
	settlementRepository *dicts.SettlementRepository
	settlementPopularity *repositories.SettlementPopularityRepository
	stationCodes         *dicts.StationCodesRepository
	logger               log.Logger
}

func NewCityListGetter(
	stationRepository *dicts.StationRepository,
	settlementRepository *dicts.SettlementRepository,
	settlementPopularity *repositories.SettlementPopularityRepository,
	stationCodes *dicts.StationCodesRepository,
	logger log.Logger,
) *CityListGetter {
	return &CityListGetter{
		stationRepository:    stationRepository,
		settlementRepository: settlementRepository,
		settlementPopularity: settlementPopularity,
		stationCodes:         stationCodes,
		logger:               logger,
	}
}

func (h *CityListGetter) GetCities(ctx context.Context, countryID int) ([]*rasp.TSettlement, bool) {
	cities := h.settlementRepository.GetByCountryID(countryID)
	if len(cities) == 0 {
		return nil, false
	}
	cities = h.filterCities(cities)
	cities = h.sortByPopularity(ctx, cities)
	size := CityTopNumber
	if size > len(cities) {
		size = len(cities)
	}
	cities = cities[:size]
	h.logger.Debugf("%d cities fetched", len(cities))
	return cities, true
}

func (h *CityListGetter) filterCities(cities []*rasp.TSettlement) []*rasp.TSettlement {
	result := []*rasp.TSettlement{}
	aviaCodes := []rasp.ECodeSystem{
		rasp.ECodeSystem_CODE_SYSTEM_ICAO,
		rasp.ECodeSystem_CODE_SYSTEM_IATA,
		rasp.ECodeSystem_CODE_SYSTEM_SIRENA,
	}
	for _, city := range cities {
		if city.IsHidden {
			continue
		}
		cityStations, ok := h.stationRepository.GetBySettlementID(city.Id)
		if !ok || len(cityStations) == 0 {
			continue
		}
		hasAviation := false
		canSearch := false
		for _, station := range cityStations {
			codes, ok := h.stationCodes.GetCodesByStationID(station.Id)
			if !ok {
				continue
			}
			for _, code := range codes {
				for _, c := range aviaCodes {
					if code.SystemId == c {
						canSearch = true
						break
					}
				}
				if canSearch {
					break
				}
			}
			hasAviation = hasAviation || (station.Type == rasp.TStation_TYPE_AIRPORT)
		}
		if !hasAviation || !canSearch {
			continue
		}
		result = append(result, city)
	}
	return result
}

type cityWithPopularity struct {
	city       *rasp.TSettlement
	popularity uint32
}

func (h *CityListGetter) sortByPopularity(ctx context.Context, cities []*rasp.TSettlement) []*rasp.TSettlement {
	var citiesInfo []cityWithPopularity
	for _, city := range cities {
		popularity, ok := h.settlementPopularity.Get(ctx, uint32(city.Id))
		if !ok {
			popularity = 0
		}
		citiesInfo = append(citiesInfo, cityWithPopularity{city, popularity})
	}
	sort.Slice(
		citiesInfo,
		func(i, j int) bool {
			return citiesInfo[i].popularity > citiesInfo[j].popularity
		},
	)
	var result []*rasp.TSettlement
	for _, cityInfo := range citiesInfo {
		result = append(result, cityInfo.city)
	}
	return result
}
