package handler

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"

	"github.com/go-chi/chi/v5"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/countrysearch"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/pkg/dicts"
	"a.yandex-team.ru/travel/library/go/httputil"
)

type notFoundError struct{}

func (n *notFoundError) Error() string {
	return "not found"
}

type countryRequest struct {
	CountryID int `json:"countryId"`
}

type cityInfo struct {
	CityID int32 `json:"cityId"`
}

type countryResponse struct {
	Cities []cityInfo `json:"cities"`
}

type CountrySearchHandler struct {
	registry   *dicts.Registry
	cityGetter *countrysearch.CityListGetter
	logger     log.Logger
}

func NewCountrySearchHandler(
	registry *dicts.Registry,
	cityGetter *countrysearch.CityListGetter,
	logger log.Logger,
) *CountrySearchHandler {
	return &CountrySearchHandler{
		registry:   registry,
		cityGetter: cityGetter,
		logger:     logger,
	}
}

func (h *CountrySearchHandler) GetRouteBuilder() func(r chi.Router) {
	return func(r chi.Router) {
		r.Get("/country_search/popular_cities_desc", h.GetCities)
	}
}

func (h *CountrySearchHandler) GetCities(writer http.ResponseWriter, request *http.Request) {
	span, _ := opentracing.StartSpanFromContext(
		request.Context(),
		"internal.handler.CountrySearchHandler: PopularCities",
	)
	defer span.Finish()

	r, err := h.parseRequest(request)
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, writer)
		return
	}
	result, err := h.buildResponse(request.Context(), r)
	if err != nil {
		switch err.(type) {
		case *notFoundError:
			httputil.HandleError(err, http.StatusNotFound, writer)
		default:
			httputil.HandleError(err, http.StatusInternalServerError, writer)
		}
		return
	}
	serialized, err := json.Marshal(result)
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, writer)
		return
	}
	writer.WriteHeader(http.StatusOK)
	writer.Header().Set("Content-Type", "application/json")
	_, err = writer.Write(serialized)
	if err != nil {
		h.logger.Errorf("some error during writing a response: %v", err)
	}
}

func (h *CountrySearchHandler) parseRequest(request *http.Request) (*countryRequest, error) {
	query := request.URL.Query()
	value, ok := query["country_id"]
	if !ok || len(value) < 1 {
		return nil, fmt.Errorf("no argument country_id")
	}
	countryID, err := strconv.Atoi(value[0])
	if err != nil {
		return nil, err
	}
	return &countryRequest{CountryID: countryID}, nil
}

func (h *CountrySearchHandler) buildResponse(ctx context.Context, r *countryRequest) (*countryResponse, error) {
	citiesInfo, err := h.getCities(ctx, r)
	if err != nil {
		return nil, err
	}
	return &countryResponse{
		Cities: citiesInfo,
	}, nil
}

func (h *CountrySearchHandler) getCities(ctx context.Context, r *countryRequest) ([]cityInfo, error) {
	cities, found := h.cityGetter.GetCities(ctx, r.CountryID)
	if !found {
		return nil, &notFoundError{}
	}
	h.logger.Debugf("%d cities fetched", len(cities))

	citiesInfo := []cityInfo{}
	for _, city := range cities {
		citiesInfo = append(citiesInfo, cityInfo{
			CityID: city.Id,
		})
	}

	return citiesInfo, nil
}
