package landingcityto

import (
	"context"
	"fmt"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
)

type CityToRouteCrosslinksGetter func(context.Context, uint32, string) ([]CityToRouteCrosslink, error)
type CityToMonthAndYearPriceInfoGetter func(context.Context, uint32, string) (*CityToMonthAndYearPriceInfo, error)
type CityToNearestCitiesGetter func(context.Context, uint32, string, CityToRouteCrosslinksGetter) ([]CityToNearestCityWithPrice, error)

type landingCityToFieldBuilder func(ctx context.Context, landing *CityToLanding, query *CityToQuery) error

type Service struct {
	logger         log.Logger
	fieldsBuilders map[string]landingCityToFieldBuilder

	cityToRouteCrosslinksGetter       CityToRouteCrosslinksGetter
	cityToMonthAndYearPriceInfoGetter CityToMonthAndYearPriceInfoGetter
	cityToNearestCitiesGetter         CityToNearestCitiesGetter
}

func (s *Service) Build(ctx context.Context, query *CityToQuery) (
	result *CityToLanding, err error,
) {
	span, ctx := opentracing.StartSpanFromContext(ctx, "internal.landingcityto.Service:Build")
	defer span.Finish()

	landing := &CityToLanding{ToID: query.ToID}
	emptyLanding := true
	if len(query.Fields) > 0 {
		for _, field := range query.Fields {
			if builder, ok := s.fieldsBuilders[field]; ok {
				err = builder(ctx, landing, query)
				emptyLanding = emptyLanding && err != nil
			}
		}
	} else {
		for field := range s.fieldsBuilders {
			err = s.fieldsBuilders[field](ctx, landing, query)
			emptyLanding = emptyLanding && err != nil
		}
	}
	if emptyLanding {
		return nil, ErrorNotFound
	}
	return landing, nil
}

func (s *Service) buildCityToRouteCrosslinks(ctx context.Context, landing *CityToLanding, query *CityToQuery) error {
	if cityToRouteCrosslinks, err := s.cityToRouteCrosslinksGetter(ctx, query.ToID, query.NationalVersion); err != nil {
		return fmt.Errorf("no city route crosslinks for query: %+v", query)
	} else {
		landing.CityToRouteCrosslinks = cityToRouteCrosslinks
	}
	return nil
}

func (s *Service) buildCityToMonthAndYearPrices(ctx context.Context, landing *CityToLanding, query *CityToQuery) error {
	if cityToMonthAndYearPriceInfo, err := s.cityToMonthAndYearPriceInfoGetter(ctx, query.ToID, query.NationalVersion); err != nil {
		return fmt.Errorf("no city to month and year price info getter for query: %+v", query)
	} else {
		landing.CityToMonthAndYearPrices = *cityToMonthAndYearPriceInfo
	}
	return nil
}

func (s *Service) buildCityToNearestCities(ctx context.Context, landing *CityToLanding, query *CityToQuery) error {
	if cityToNearestCities, err := s.cityToNearestCitiesGetter(ctx, query.ToID, query.NationalVersion, s.cityToRouteCrosslinksGetter); err != nil {
		return fmt.Errorf("no city to nearest cities for query: %+v", query)
	} else {
		landing.CityToNearestCities = cityToNearestCities
	}
	return nil
}

func NewCityToService(
	l log.Logger,
	cityToRouteCrosslinksGetter CityToRouteCrosslinksGetter,
	cityToMonthAndYearPriceInfoGetter CityToMonthAndYearPriceInfoGetter,
	cityToNearestCitiesGetter CityToNearestCitiesGetter,
) *Service {
	service := &Service{
		logger: l,

		cityToRouteCrosslinksGetter:       cityToRouteCrosslinksGetter,
		cityToMonthAndYearPriceInfoGetter: cityToMonthAndYearPriceInfoGetter,
		cityToNearestCitiesGetter:         cityToNearestCitiesGetter,
	}
	service.fieldsBuilders = map[string]landingCityToFieldBuilder{
		cityToRouteCrosslinksField:    service.buildCityToRouteCrosslinks,
		cityToMonthAndYearPricesField: service.buildCityToMonthAndYearPrices,
		cityToNearestCitiesField:      service.buildCityToNearestCities,
	}
	return service
}

const (
	cityToRouteCrosslinksField    string = "cityToRouteCrosslinks"
	cityToMonthAndYearPricesField string = "cityToMonthAndYearPrices"
	cityToNearestCitiesField      string = "cityToNearestCities"
)
