package handler

import (
	"encoding/json"
	"net/http"

	"github.com/go-chi/chi/v5"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
	"a.yandex-team.ru/travel/library/go/httputil"
)

type httpResponse struct {
	Result *models.Landing     `json:"result"`
	Error  *httputil.HTTPError `json:"error,omitempty"`
}

type RoutesProvider interface {
	GetAll() []models.Route
}

type HTTPLandingRoutesHandler struct {
	routesProvider RoutesProvider
	logger         log.Logger
}

func NewHTTPLandingRoutesHandler(routesProvider RoutesProvider, logger log.Logger) *HTTPLandingRoutesHandler {
	return &HTTPLandingRoutesHandler{routesProvider: routesProvider, logger: logger}
}

func (h *HTTPLandingRoutesHandler) GetAll(w http.ResponseWriter, r *http.Request) {
	span, _ := opentracing.StartSpanFromContext(r.Context(), "internal.handler.HTTPLandingRoutesHandler: GetAll")
	defer span.Finish()

	routes := h.routesProvider.GetAll()
	resultBytes, err := json.Marshal(routes)
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	w.WriteHeader(http.StatusOK)
	w.Header().Set("Content-Type", "application/json")
	_, _ = w.Write(resultBytes)
}

func (h *HTTPLandingRoutesHandler) GetRouteBuilder() func(r chi.Router) {
	return func(r chi.Router) {
		r.Get("/landing/all_routes/", h.GetAll)
	}
}
