package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type AlternativeRoutesPricesRepository struct {
	alternativeRoutesPricesByRoute atomic.Value
	dataGetter                     func() ([]models.AlternativeRoutePrice, error)
	logger                         log.Logger
}

func NewAlternativeRoutesPricesRepository(
	logger log.Logger,
	dataGetter func() ([]models.AlternativeRoutePrice, error),
) *AlternativeRoutesPricesRepository {
	return &AlternativeRoutesPricesRepository{dataGetter: dataGetter, logger: logger}
}

func (r *AlternativeRoutesPricesRepository) Update() error {
	r.logger.Info("precaching alternative_routes_prices")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var alternativeRoutesPricesByRoute = make(map[models.Route][]models.AlternativeRoutePrice)
	for _, alternativeRoutePrice := range data {
		key := models.Route{FromID: alternativeRoutePrice.FromID, ToID: alternativeRoutePrice.ToID, NationalVersion: alternativeRoutePrice.NationalVersion}
		alternativeRoutesPricesByRoute[key] = append(alternativeRoutesPricesByRoute[key], alternativeRoutePrice)
	}
	r.alternativeRoutesPricesByRoute.Store(alternativeRoutesPricesByRoute)
	r.logger.Info("alternative_routes_prices precached")
	return nil
}

func (r *AlternativeRoutesPricesRepository) Get(ctx context.Context, fromID, toID uint32, nationalVersion string) (
	[]models.AlternativeRoutePrice,
	error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.AlternativeRoutesPricesRepository:Get")
	defer span.Finish()

	d := models.Route{FromID: fromID, ToID: toID, NationalVersion: nationalVersion}
	if value, found := r.alternativeRoutesPricesByRoute.Load().(map[models.Route][]models.AlternativeRoutePrice)[d]; found {
		return value, nil
	}
	return nil, models.ErrorNotFound
}
