package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/landingcityto"
)

type CityToMonthAndYearPricesRepository struct {
	cityToMonthAndYearPrices atomic.Value
	dataGetter               func() ([]landingcityto.CityToMonthAndYearPriceInfo, error)
	logger                   log.Logger
}

func NewCityToMonthAndYearPricesRepository(
	logger log.Logger,
	dataGetter func() ([]landingcityto.CityToMonthAndYearPriceInfo, error),
) *CityToMonthAndYearPricesRepository {
	return &CityToMonthAndYearPricesRepository{dataGetter: dataGetter, logger: logger}
}

func (r *CityToMonthAndYearPricesRepository) Update() error {
	r.logger.Info("precaching city_to_month_and_year_prices")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var pricesByCityTo = make(map[landingcityto.CityTo]landingcityto.CityToMonthAndYearPriceInfo)
	for _, cityToPrices := range data {
		key := landingcityto.CityTo{ToID: cityToPrices.ToID, NationalVersion: cityToPrices.NationalVersion}
		pricesByCityTo[key] = cityToPrices
	}
	r.cityToMonthAndYearPrices.Store(pricesByCityTo)
	r.logger.Info("city_to_month_and_year_prices precached")
	return nil
}

func (r *CityToMonthAndYearPricesRepository) Get(ctx context.Context, toID uint32, nationalVersion string) (
	*landingcityto.CityToMonthAndYearPriceInfo,
	error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.CityToMonthAndYearPricesRepository:Get")
	defer span.Finish()

	d := landingcityto.CityTo{ToID: toID, NationalVersion: nationalVersion}
	if value, found := r.cityToMonthAndYearPrices.Load().(map[landingcityto.CityTo]landingcityto.CityToMonthAndYearPriceInfo)[d]; found {
		return &value, nil
	}
	return nil, landingcityto.ErrorNotFound
}
