package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/landingcityto"
)

type CityToNearestCitiesRepository struct {
	cityToNearestCities atomic.Value
	dataGetter          func() ([]landingcityto.CityToNearestCities, error)
	logger              log.Logger
}

func NewCityToNearestCitiesRepository(
	logger log.Logger,
	dataGetter func() ([]landingcityto.CityToNearestCities, error),
) *CityToNearestCitiesRepository {
	return &CityToNearestCitiesRepository{dataGetter: dataGetter, logger: logger}
}

func (r *CityToNearestCitiesRepository) Update() error {
	r.logger.Info("precaching city_to_nearest_cities")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var nearestByCityTo = make(map[landingcityto.CityTo]landingcityto.CityToNearestCities)
	for _, record := range data {
		key := landingcityto.CityTo{ToID: record.ToID, NationalVersion: record.NationalVersion}
		nearestByCityTo[key] = record
	}
	r.cityToNearestCities.Store(nearestByCityTo)
	r.logger.Info("city_to_nearest_cities precached")
	return nil
}

func (r *CityToNearestCitiesRepository) Get(
	ctx context.Context,
	toID uint32,
	nationalVersion string,
	crosslinksGetter landingcityto.CityToRouteCrosslinksGetter,
) (
	[]landingcityto.CityToNearestCityWithPrice,
	error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.CityToNearestCitiesRepository:Get")
	defer span.Finish()

	d := landingcityto.CityTo{ToID: toID, NationalVersion: nationalVersion}
	value, found := r.cityToNearestCities.Load().(map[landingcityto.CityTo]landingcityto.CityToNearestCities)[d]
	if !found || len(value.NearestCityIds) == 0 {
		return nil, landingcityto.ErrorNotFound
	}

	ids := value.NearestCityIds
	nearestCities := make([]landingcityto.CityToNearestCityWithPrice, 0, len(ids))
	for _, id := range ids {
		crosslinks, err := crosslinksGetter(ctx, id, nationalVersion)
		if err != nil {
			continue
		}
		if len(crosslinks) > 0 {
			min := findCrosslinkWithMinPrice(crosslinks)
			nearestCities = append(nearestCities, landingcityto.CityToNearestCityWithPrice{
				ID:       id,
				Price:    min.Price,
				Currency: min.Currency,
				Date:     min.Date,
			})
		} else {
			nearestCities = append(nearestCities, landingcityto.CityToNearestCityWithPrice{
				ID:       id,
				Price:    nil,
				Currency: nil,
				Date:     nil,
			})
		}
	}
	return nearestCities, nil
}

func findCrosslinkWithMinPrice(crosslinks []landingcityto.CityToRouteCrosslink) landingcityto.CityToRouteCrosslink {
	min := crosslinks[0]
	for _, c := range crosslinks {
		if c.Price != nil {
			if min.Price == nil {
				min = c
			} else {
				if *min.Price > *c.Price {
					min = c
				}
			}
		}
	}
	return min
}
