package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/landingcityto"
)

type CityToRouteCrosslinksRepository struct {
	cityToRouteCrosslinksByCityTo atomic.Value
	dataGetter                    func() ([]landingcityto.CityToRouteCrosslink, error)
	logger                        log.Logger
}

func NewCityToRouteCrosslinksRepository(
	logger log.Logger,
	dataGetter func() ([]landingcityto.CityToRouteCrosslink, error),
) *CityToRouteCrosslinksRepository {
	return &CityToRouteCrosslinksRepository{dataGetter: dataGetter, logger: logger}
}

func (r *CityToRouteCrosslinksRepository) Update() error {
	r.logger.Info("precaching city_to_route_crosslinks")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var cityToRouteCrosslinksByCityTo = make(map[landingcityto.CityTo][]landingcityto.CityToRouteCrosslink)
	for _, cityToRouteCrosslink := range data {
		key := landingcityto.CityTo{
			ToID:            cityToRouteCrosslink.ToID,
			NationalVersion: cityToRouteCrosslink.NationalVersion,
		}
		cityToRouteCrosslinksByCityTo[key] = append(cityToRouteCrosslinksByCityTo[key], cityToRouteCrosslink)
	}
	r.cityToRouteCrosslinksByCityTo.Store(cityToRouteCrosslinksByCityTo)
	r.logger.Info("city_to_route_crosslinks precached")
	return nil
}

func (r *CityToRouteCrosslinksRepository) Get(ctx context.Context, toID uint32, nationalVersion string) (
	[]landingcityto.CityToRouteCrosslink,
	error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.CityToRouteCrosslinksRepository:Get")
	defer span.Finish()

	d := landingcityto.CityTo{ToID: toID, NationalVersion: nationalVersion}
	if value, found := r.cityToRouteCrosslinksByCityTo.Load().(map[landingcityto.CityTo][]landingcityto.CityToRouteCrosslink)[d]; found {
		return value, nil
	}
	return nil, landingcityto.ErrorNotFound
}
