package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type MedianPricesRepository struct {
	medianPricesByRoute atomic.Value
	dataGetter          func() ([]models.MedianPrices, error)
	logger              log.Logger
}

func NewMedianPricesRepository(logger log.Logger, dataGetter func() ([]models.MedianPrices, error)) *MedianPricesRepository {
	return &MedianPricesRepository{dataGetter: dataGetter, logger: logger}
}

func (r *MedianPricesRepository) Update() error {
	r.logger.Info("precaching median_prices")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var medianPricesByRoute = make(map[models.Route]models.MedianPrices)
	for _, medianPrices := range data {
		key := models.Route{FromID: medianPrices.FromID, ToID: medianPrices.ToID, NationalVersion: medianPrices.NationalVersion}
		medianPricesByRoute[key] = medianPrices
	}
	r.medianPricesByRoute.Store(medianPricesByRoute)
	r.logger.Info("median_prices precached")
	return nil
}

func (r *MedianPricesRepository) Get(ctx context.Context, fromID, toID uint32, nationalVersion string) (
	*models.MedianPrices,
	error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.MedianPricesRepository:Get")
	defer span.Finish()

	d := models.Route{FromID: fromID, ToID: toID, NationalVersion: nationalVersion}
	if value, found := r.medianPricesByRoute.Load().(map[models.Route]models.MedianPrices)[d]; found {
		return &value, nil
	}
	return nil, models.ErrorNotFound
}
