package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type MinPricesByAirlineRepository struct {
	minPriceByRouteAndAirline atomic.Value
	dataGetter                func() ([]models.MinPriceByAirline, error)
	logger                    log.Logger
}

func NewMinPricesByAirlineRepository(
	logger log.Logger,
	dataGetter func() ([]models.MinPriceByAirline, error),
) *MinPricesByAirlineRepository {
	return &MinPricesByAirlineRepository{dataGetter: dataGetter, logger: logger}
}

type routeWithAirlineID struct {
	route     models.Route
	airlineID uint32
}

func (r *MinPricesByAirlineRepository) Update() error {
	r.logger.Info("precaching min_prices_by_airline")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var minPriceByRouteAndAirline = make(map[routeWithAirlineID]models.MinPriceByAirline)
	for _, minPriceByAirline := range data {
		route := models.Route{
			FromID:          minPriceByAirline.FromID,
			ToID:            minPriceByAirline.ToID,
			NationalVersion: minPriceByAirline.NationalVersion,
		}
		key := routeWithAirlineID{route: route, airlineID: minPriceByAirline.AirlineID}
		minPriceByRouteAndAirline[key] = minPriceByAirline
	}
	r.minPriceByRouteAndAirline.Store(minPriceByRouteAndAirline)
	r.logger.Info("min_prices_by_airline precached")
	return nil
}

func (r *MinPricesByAirlineRepository) Get(
	ctx context.Context,
	fromID, toID uint32,
	nationalVersion string,
	airlineID uint32,
) (
	*models.MinPriceByAirline,
	error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.MinPricesByAirlineRepository:Get")
	defer span.Finish()

	route := models.Route{FromID: fromID, ToID: toID, NationalVersion: nationalVersion}
	key := routeWithAirlineID{route: route, airlineID: airlineID}
	if value, found := r.minPriceByRouteAndAirline.Load().(map[routeWithAirlineID]models.MinPriceByAirline)[key]; found {
		return &value, nil
	}
	return nil, models.ErrorNotFound
}
