package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type PopularMonthsRepository struct {
	popularMonthByRoute atomic.Value
	dataGetter          func() ([]models.PopularMonth, error)
	logger              log.Logger
}

func NewPopularMonthsRepository(logger log.Logger, dataGetter func() ([]models.PopularMonth, error)) *PopularMonthsRepository {
	return &PopularMonthsRepository{dataGetter: dataGetter, logger: logger}
}

func (r *PopularMonthsRepository) Update() error {
	r.logger.Info("precaching popular_months")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var popularMonthByRoute = make(map[models.Route]models.PopularMonth)
	for _, popularMonth := range data {
		key := models.Route{FromID: popularMonth.FromID, ToID: popularMonth.ToID, NationalVersion: popularMonth.NationalVersion}
		popularMonthByRoute[key] = popularMonth
	}
	r.popularMonthByRoute.Store(popularMonthByRoute)
	r.logger.Info("popular_months precached")
	return nil
}

func (r *PopularMonthsRepository) Get(ctx context.Context, fromID, toID uint32, nationalVersion string) (*models.PopularMonth, error) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.PopularMonthsRepository:Get")
	defer span.Finish()

	d := models.Route{FromID: fromID, ToID: toID, NationalVersion: nationalVersion}
	if value, found := r.popularMonthByRoute.Load().(map[models.Route]models.PopularMonth)[d]; found {
		return &value, nil
	}
	return nil, models.ErrorNotFound
}
