package repositories

import (
	"reflect"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/pkg/tables"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/pkg/updating"
)

type Registry struct {
	PopularMonthsRepository      *PopularMonthsRepository
	ReturnTicketPricesRepository *ReturnTicketPricesRepository
	TopAirlinesRepository        *TopAirlinesRepository
	AlternativeRoutesPrices      *AlternativeRoutesPricesRepository
	MedianPrices                 *MedianPricesRepository
	RouteInfos                   *RouteInfosRepository
	MinPricesByAirline           *MinPricesByAirlineRepository
	RouteCrosslinks              *RouteCrosslinksRepository
	Routes                       *RoutesRepository
	CityToRouteCrosslinks        *CityToRouteCrosslinksRepository
	CityToMonthAndYearPrices     *CityToMonthAndYearPricesRepository
	CityToNearestCities          *CityToNearestCitiesRepository
	SettlementPopularity         *SettlementPopularityRepository
}

type periodicallyUpdatable interface {
	Update() error
}

type Config struct {
	UpdatePeriod time.Duration
}

var DefaultConfig = Config{
	UpdatePeriod: time.Hour,
}

func NewRegistry(config Config, tablesRegistry *tables.Registry, logger log.Logger) (r *Registry, err error) {
	r = &Registry{
		PopularMonthsRepository:      NewPopularMonthsRepository(logger, tablesRegistry.PopularMonths.GetAll),
		ReturnTicketPricesRepository: NewReturnTicketPricesRepository(logger, tablesRegistry.ReturnTicketPrices.GetAll),
		TopAirlinesRepository:        NewTopAirlinesRepository(logger, tablesRegistry.TopAirlines.GetAll),
		AlternativeRoutesPrices:      NewAlternativeRoutesPricesRepository(logger, tablesRegistry.AlternativeRoutesPrices.GetAll),
		MedianPrices:                 NewMedianPricesRepository(logger, tablesRegistry.MedianPrices.GetAll),
		RouteInfos:                   NewRouteInfosRepository(logger, tablesRegistry.RouteInfos.GetAll),
		MinPricesByAirline:           NewMinPricesByAirlineRepository(logger, tablesRegistry.MinPricesByAirline.GetAll),
		RouteCrosslinks:              NewRouteCrosslinksRepository(logger, tablesRegistry.RouteCrosslinks.GetAll),
		Routes:                       NewRoutesRepository(logger, tablesRegistry.Routes.GetAll),
		CityToRouteCrosslinks:        NewCityToRouteCrosslinksRepository(logger, tablesRegistry.CityToRouteCrosslinks.GetAll),
		CityToMonthAndYearPrices:     NewCityToMonthAndYearPricesRepository(logger, tablesRegistry.CityToMonthAndYearPrices.GetAll),
		CityToNearestCities:          NewCityToNearestCitiesRepository(logger, tablesRegistry.CityToNearestCities.GetAll),
		SettlementPopularity:         NewSettlementPopularityRepository(logger, tablesRegistry.SettlementPopularity.GetAll),
	}

	periodicallyUpdatable := []periodicallyUpdatable{
		r.ReturnTicketPricesRepository,
		r.PopularMonthsRepository,
		r.TopAirlinesRepository,
		r.AlternativeRoutesPrices,
		r.MedianPrices,
		r.RouteInfos,
		r.MinPricesByAirline,
		r.RouteCrosslinks,
		r.Routes,
		r.CityToRouteCrosslinks,
		r.CityToMonthAndYearPrices,
		r.CityToNearestCities,
		r.SettlementPopularity,
	}
	for _, repository := range periodicallyUpdatable {
		err = repository.Update()
		if err != nil {
			return nil, err
		}
	}
	updater := updating.NewBackgroundUpdater(logger)
	for _, u := range periodicallyUpdatable {
		updater.RunBackgroundUpdates(updating.TypeToString(reflect.TypeOf(u)), u.Update, config.UpdatePeriod)
	}

	return r, nil
}
