package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type ReturnTicketPricesRepository struct {
	returnTicketPriceByRoute atomic.Value
	dataGetter               func() ([]models.ReturnTicketPrice, error)
	logger                   log.Logger
}

func NewReturnTicketPricesRepository(
	logger log.Logger,
	dataGetter func() ([]models.ReturnTicketPrice, error),
) *ReturnTicketPricesRepository {
	return &ReturnTicketPricesRepository{dataGetter: dataGetter, logger: logger}
}

func (r *ReturnTicketPricesRepository) Update() error {
	r.logger.Info("precaching return_ticket_prices")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var returnTicketPriceByRoute = make(map[models.Route]models.ReturnTicketPrice)
	for _, returnTicketPrice := range data {
		key := models.Route{FromID: returnTicketPrice.FromID, ToID: returnTicketPrice.ToID, NationalVersion: returnTicketPrice.NationalVersion}
		returnTicketPriceByRoute[key] = returnTicketPrice
	}
	r.returnTicketPriceByRoute.Store(returnTicketPriceByRoute)
	r.logger.Info("return_ticket_prices precached")
	return nil
}

func (r *ReturnTicketPricesRepository) Get(ctx context.Context, fromID, toID uint32, nationalVersion string) (
	*models.ReturnTicketPrice,
	error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.ReturnTicketPricesRepository:Get")
	defer span.Finish()

	d := models.Route{FromID: fromID, ToID: toID, NationalVersion: nationalVersion}
	if value, found := r.returnTicketPriceByRoute.Load().(map[models.Route]models.ReturnTicketPrice)[d]; found {
		return &value, nil
	}
	return nil, models.ErrorNotFound
}
