package repositories

import (
	"context"
	"sync/atomic"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/avia_statistics/api/internal/seolanding/models"
)

type RouteCrosslinksRepository struct {
	routeCrosslinksByRoute atomic.Value
	dataGetter             func() ([]models.RouteCrosslink, error)
	logger                 log.Logger
}

func NewRouteCrosslinksRepository(
	logger log.Logger,
	dataGetter func() ([]models.RouteCrosslink, error),
) *RouteCrosslinksRepository {
	return &RouteCrosslinksRepository{dataGetter: dataGetter, logger: logger}
}

func (r *RouteCrosslinksRepository) Update() error {
	r.logger.Info("precaching route_crosslinks")
	data, err := r.dataGetter()
	if err != nil {
		return err
	}
	var routeCrosslinksByRoute = make(map[models.Route][]models.RouteCrosslink)
	for _, routeCrosslink := range data {
		key := models.Route{
			FromID:          routeCrosslink.FromID,
			ToID:            routeCrosslink.ToID,
			NationalVersion: routeCrosslink.NationalVersion,
		}
		routeCrosslinksByRoute[key] = append(routeCrosslinksByRoute[key], routeCrosslink)
	}
	r.routeCrosslinksByRoute.Store(routeCrosslinksByRoute)
	r.logger.Info("route_crosslinks precached")
	return nil
}

func (r *RouteCrosslinksRepository) Get(ctx context.Context, fromID, toID uint32, nationalVersion string) (
	[]models.RouteCrosslink,
	error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.storage.RouteCrosslinksRepository:Get")
	defer span.Finish()

	d := models.Route{FromID: fromID, ToID: toID, NationalVersion: nationalVersion}
	if value, found := r.routeCrosslinksByRoute.Load().(map[models.Route][]models.RouteCrosslink)[d]; found {
		return value, nil
	}
	return nil, models.ErrorNotFound
}
